/*!
 * @file fastec_sensor.cpp
 * @brief This is the javascript object responsible for housing
 *        different sensor related record setting calculations
 *
 * Copyright 2014 (c) Fastec Imaging as an unpublished work.
 * All Rights Reserved.
 *
 * The information contained herein is confidential property of
 * Fastec Imaging. The use, copying, transfer or disclosure of such
 * information is prohibited except by express written agreement
 * with Fastec Imaging.
 */

// Hex value for 'AL5M' and 'AL5C' Anafocus Sensor Model types
var gLINCE5MMono = "414C354D";
var gLINCE5Color = "414C3543";

// Possible bitmode settings for sensors.
var GIGE_BITMODE_LOWER_8_BITS = 0x00; //!< Lower 8 bits of sensor data
var GIGE_BITMODE_MIDDLE_8_BITS = 0x01; //!< Middle 8 bits of sensor data
var GIGE_BITMODE_UPPER_8_BITS = 0x02; //!< Upper 8 bits of sensor data
var GIGE_BITMODE_10_BITS = 0x03; //!< All 10 bits of sensor data for 10-bit sensor
var GIGE_BITMODE_7_0 = 0x10; //!< Bits 7:0
var GIGE_BITMODE_8_1 = 0x11; //!< Bits 8:1
var GIGE_BITMODE_9_2 = 0x12; //!< Bits 9:2
var GIGE_BITMODE_10_3 = 0x13; //!< Bits 10:3
var GIGE_BITMODE_11_4 = 0x14; //!< Bits 11:4
var GIGE_BITMODE_9_0 = 0x20; //!< Bits 9:0
var GIGE_BITMODE_10_1 = 0x21; //!< Bits 10:1
var GIGE_BITMODE_11_2 = 0x23; //!< Bits 11:2
var GIGE_BITMODE_11_0 = 0x30; //!< Bits 11:0

//--------------------------------------
// Is the Camera bitmode set to 8 bits?
//--------------------------------------
function IS_8_BIT_MODE(bit_mode) {
    if (
        bit_mode <= GIGE_BITMODE_UPPER_8_BITS ||
        (GIGE_BITMODE_7_0 <= bit_mode && bit_mode <= GIGE_BITMODE_11_4)
    ) {
        return true;
    }

    return false;
}

//--------------------------------------
// Is the Camera bitmode set to 10 bits?
//--------------------------------------
function IS_10_BIT_MODE(bit_mode) {
    if (
        bit_mode == GIGE_BITMODE_10_BITS ||
        (GIGE_BITMODE_9_0 <= bit_mode && bit_mode <= GIGE_BITMODE_11_2)
    ) {
        return true;
    }
    return false;
}

//--------------------------------------
// Is the Camera bitmode set to 12 bits?
//--------------------------------------
function IS_12_BIT_MODE(bit_mode) {
    if (bit_mode == GIGE_BITMODE_11_0) {
        return true;
    }
    return false;
}

//--------------------------------------------------------------------
// sensor_max_frame_rate
//
// determines the sensor-specific maximum frame rate for a given ROI,
// bitmode and sensor options
//
// this is an interface to sort out type of sensor and forwards call
// to the appropriate implementation
//--------------------------------------------------------------------
function sensor_max_frame_rate(
    roiX,
    roiY,
    roiWidth,
    roiHeight,
    bitMode,
    sensopts
) {
    // depending on the sensor pull out the frame rate
    var maxFrameRate = 0; // not set yet.

    // if lince sensor call it's calculator for maximum frame rate
    if (gCameraData.isLINCESensor()) {
        maxFrameRate = lince5m_max_fps_setting(
            roiX,
            roiY,
            roiWidth,
            roiHeight,
            bitMode,
            sensopts
        );
    }
    return maxFrameRate;
}

//--------------------------------------------------------------------
// sensor_fpn_pixel_support
//
// determines the sensor-specific state of supporting the fpn pixel
//--------------------------------------------------------------------
function sensor_fpn_pixel_support() {
    // if lince sensor call it's calculator for maximum frame rate
    if (gCameraData.isLINCESensor()) {
        return lince_fpn_pixel_support();
    }
    return false;
}

//--------------------------------------------------------------------
// sensor_get_max_res_width
//
// determines the sensor-specific state of supporting the fpn pixel
//--------------------------------------------------------------------
function sensor_max_res_width(bit_mode) {
    var sensorMaxResWidth = 1;
    // if lince sensor call it's calculator for maximum frame rate
    if (gCameraData.isLINCESensor()) {
        sensorMaxResWidth = lince_max_res_width(bit_mode);
    }
    return sensorMaxResWidth;
}

//--------------------------------------------------------------------
// sensor_max_res_height
//
// determines the sensor-specific state of supporting the fpn pixel
//--------------------------------------------------------------------
function sensor_max_res_height(bit_mode) {
    var sensorMaxResHeight = 1;
    // if lince sensor call it's calculator for maximum frame rate
    if (gCameraData.isLINCESensor()) {
        sensorMaxResHeight = lince_max_res_height(bit_mode);
    }
    return sensorMaxResHeight;
}

//--------------------------------------------------------------------
// the result object for function sensor_max_roi_result
//--------------------------------------------------------------------
function sensor_max_roi_result() {
    // object to hold return value for sensor_max_roi_result
    this.x = 0;
    this.y = 0;
    this.max_width = 0;
    this.max_height = 0;
}
//--------------------------------------------------------------------
// sensor_max_res_height
//
// determines the sensor-specific state of supporting the fpn pixel
//--------------------------------------------------------------------
function sensor_max_roi(bit_mode) {
    // create the return ROI Object
    var retObj = new sensor_max_roi_result();
    retObj.x = 0;
    retObj.y = 0;
    retObj.max_width = sensor_max_res_width(bit_mode);
    retObj.max_height = sensor_max_res_height(bit_mode);

    return retObj;
}
