//-------------------------------------------------
// ts3_vidplay.js
//
// Copyright 2010 (c) Fastec Imaging as an unpublished work.
// All Rights Reserved.
//
// The information contained herein is confidential property of
// Fastec Imaging. The use, copying, transfer or disclosure of such
// information is prohibited except by express written agreement
// with Fastec Imaging.
//
// Fastec Imaging Trouble Shooter 3 Web Application
// Video Review -- handles the data
// exchange and management of camera data for
// the video review panel
//-------------------------------------------------

//------------------------------------
// The Fastec Player Button Object
//
// This object is responsible for managing
// the state of a player button.
//------------------------------------
function fastecPlayerBtn() {
    try {
        // Save the JQuery Selector ID for this button.
        this.thePBSelector = 0;

        // Save the attribute id for fast comparisons.
        this.theAttrID = " ";

        // A click handler for this button.
        this.clickHandlerFunc = null;

        // Stores as bound data to our object instance
        this.idData = { fastecPlayerBtn: this };

        //---------------------------
        //---------------------------
        this.onPBClick = function(e) {
            if (e.data.fastecPlayerBtn != undefined) {
                var me = e.data.fastecPlayerBtn;

                fastecLogToConsole(gLogAll, "onPBClick() - id=" + me.theAttrID);

                // Do we have a click handler?
                if (me.clickHandlerFunc !== null) {
                    // Call the click handler.
                    me.clickHandlerFunc();
                }
            }
        };

        //---------------------------
        //---------------------------
        this.onPBMouseOver = function(e) {
            if (e.data.fastecPlayerBtn != undefined) {
                var me = e.data.fastecPlayerBtn;

                fastecLogToConsole(
                    gLogAll,
                    "onPBMouseOver() - id=" + me.theAttrID
                );
            }
        };

        //---------------------------
        //---------------------------
        this.onPBMouseOut = function(e) {
            if (e.data.fastecPlayerBtn != undefined) {
                var me = e.data.fastecPlayerBtn;

                fastecLogToConsole(
                    gLogAll,
                    "onPBMouseOut() - id=" + me.theAttrID
                );
            }
        };

        //---------------------------
        //---------------------------
        this.onPBMouseDown = function(e) {
            if (e.data.fastecPlayerBtn != undefined) {
                var me = e.data.fastecPlayerBtn;

                fastecLogToConsole(
                    gLogAll,
                    "onPBMouseDown() - id=" + me.theAttrID
                );
            }
        };

        //---------------------------
        //---------------------------
        this.onPBMouseUp = function(e) {
            if (e.data.fastecPlayerBtn != undefined) {
                var me = e.data.fastecPlayerBtn;

                fastecLogToConsole(
                    gLogAll,
                    "onPBMouseUp() - id=" + me.theAttrID
                );
            }
        };

        //---------------------------
        // Set the JQuery Selector
        //---------------------------
        this.setPBSelector = function(inPBSelector) {
            this.thePBSelector = inPBSelector;
            this.theAttrID = this.thePBSelector.attr("id");
            this.thePBSelector.bind("click", this.idData, this.onPBClick);
            this.thePBSelector.bind(
                "mouseover",
                this.idData,
                this.onPBMouseOver
            );
            this.thePBSelector.bind("mouseout", this.idData, this.onPBMouseOut);
            this.thePBSelector.bind(
                "mousedown",
                this.idData,
                this.onPBMouseDown
            );
            this.thePBSelector.bind("mouseup", this.idData, this.onPBMouseUp);
        };

        //---------------------------
        // Set the button visible state
        //---------------------------
        this.setVisibleState = function(amIVisible) {
            if (amIVisible) {
                this.thePBSelector.show();
            } else {
                this.thePBSelector.hide();
            }
        };

        //---------------------------
        // Set the click handler for this button
        //---------------------------
        this.setClickHandler = function(theClickHandlerFunc) {
            this.clickHandlerFunc = theClickHandlerFunc;
            this.clickHandlerExists = true;
        };
    } catch (err) {
        var theDetails = "caught: fastecPlayerBtn -" + err.description;
        gFaultHandler.logError(theDetails);
    }
}

//-----------------------------------------------------------------
// FIP:  PLAYER CONTROL BUTTON NUMBER IDs
//-----------------------------------------------------------------
var kfipPlayBtnID = {
    FIP_None: 0,
    FIP_GotoFirst: 1,
    FIP_RevSkip: 2,
    FIP_RevPlay: 3,
    FIP_Stop: 4,
    FIP_FwdPlay: 5,
    FIP_FwdSkip: 6,
    FIP_GotoLast: 7,
    FIP_Trigger: 8
};

//-----------------------------------------------------------------
// FIP:  PLAYER CONTROL STATE OBJECT
//-----------------------------------------------------------------
// Object is responsible for creating and
// managing the player controls.
//-----------------------------------------------------------------
function fipPlayerControls() {
    try {
        var playerCtrlStates = {
            playerStopped: 0,
            playerFwdPlay: 1,
            playerRevPlay: 2,
            playerFwdSkip: 3,
            playerRevSkip: 4
        };
        this.playerCtrlState = playerCtrlStates.playerStopped;

	// Create each of the player button objects.
	this.gotoFirst = new fastecPlayerBtn();
	this.skipRev = new fastecPlayerBtn();
	this.playRev = new fastecPlayerBtn();
	this.stop = new fastecPlayerBtn();
	this.playFwd = new fastecPlayerBtn();
	this.skipFwd = new fastecPlayerBtn();
	this.gotoLast = new fastecPlayerBtn();
	this.gotoTrig = new fastecPlayerBtn();

        //--------------------------------------
        // Initialize UI Widgets in Use
        //--------------------------------------
        this.initUI = function() {
	    // Initialize our buttons.
	    this.gotoFirst.setPBSelector($("#gotoFirstPBID"));
	    this.skipRev.setPBSelector($("#skipRevPBID"));
	    this.playRev.setPBSelector($("#playRevPBID"));
	    this.stop.setPBSelector($("#stopPBID"));
	    this.playFwd.setPBSelector($("#playFwdPBID"));
	    this.skipFwd.setPBSelector($("#skipFwdPBID"));
	    this.gotoLast.setPBSelector($("#gotoLastPBID"));
	    this.gotoTrig.setPBSelector($("#gotoTrigPBID"));

	    this.gotoFirst.setClickHandler(this.onClickGotoFirst);
	    this.skipRev.setClickHandler(this.onClickSkipRev);
	    this.playRev.setClickHandler(this.onClickPlayRev);
	    this.stop.setClickHandler(this.onClickStop);
	    this.playFwd.setClickHandler(this.onClickPlayFwd);
	    this.skipFwd.setClickHandler(this.onClickSkipFwd);
	    this.gotoLast.setClickHandler(this.onClickGotoLast);
	    this.gotoTrig.setClickHandler(this.onClickGotoTrig);
        };

        //--------------------------------------
        // Click Handlers for Player Buttons
        //--------------------------------------
        this.onClickGotoFirst = function() {
            gVidReviewUI.displayFirstFrame();
        };
        this.onClickSkipRev = function() {
            gVidReviewUI.step(false);
        };
        this.onClickPlayRev = function() {
	    gVidReviewUI.startPlay(false);
        };
        this.onClickStop = function() {
	    gVidReviewUI.stopPlay();
        };
        this.onClickPlayFwd = function() {
	    gVidReviewUI.startPlay(true);
        };
        this.onClickSkipFwd = function() {
            gVidReviewUI.step(true);
        };
        this.onClickGotoLast = function() {
            gVidReviewUI.displayLastFrame();
        };
        this.onClickGotoTrig = function() {
	    gVidReviewUI.displayTrigFrame();
        };

    } catch (err) {
        var theDetails = "caught: fipPlayerControls-" + err.description;
        gFaultHandler.logError(theDetails);
    }
}

function fastecReviewMgr() {
    this.playerBar = new fipPlayerControls();
    this.playerBar.initUI();

    this.currentFrameID = 0;
    this.fps = 30;

    this.buttonPending = kfipPlayBtnID.FIP_None;

    //---------------------------------
    //---------------------------------
    this.cbhFrameReceived = function(frameID) {
	fastecLogToConsole(gLogAll, "cbhFrameReceived(): frame=" + frameID);

	// Display the information to the user. Note the frame number is zero
	// based, so subtract the total frame count by one for the last frame.
	var frame_msg = frameID + " / " + (gCameraData.actPartNumFrames - 1);
	$("#vrFrameID").html(frame_msg);

	// Save which frame is the current frame.
	gVidReviewUI.currentFrameID = frameID;
    };

    this.handleReviewFrame = function(frameID) {
	fastecLogToConsole(gLogAll, "handleReviewFrame(): frame=" + frameID + ", pbControl=" + gCameraData.pbEnabledRegValue + ", lsControl=" + gCameraData.pbLocalControlStream);

	// Make sure we don't collide with cbhFrameReceived when in PLAY
	// mode. Single step doesn't have an issue.
	if (gCameraData.pbLocalControlStream == gRegs.GIGE_LOCALSTREAM_PLAYBACK)
	    return;

	// Display the information to the user. Note the frame number is zero
	// based, so subtract the total frame count by one for the last frame.
	var frame_msg = frameID + " / " + (gCameraData.actPartNumFrames - 1);
	$("#vrFrameID").html(frame_msg);

	// Save which frame is the current frame.
	gVidReviewUI.currentFrameID = frameID;
    };

    //---------------------------------
    //---------------------------------
    this.hide = function() {
	// Hide the review UI.
	$("#reviewSectionID").hide();
	$("#vrFrameID").html("");
    };

    //---------------------------------
    //---------------------------------
    this.show = function() {
	// Show the review UI.
	$("#reviewSectionID").show();
    };

    this.cbhCommandComplete = function(requestResult) {
	var my_msg = "cbhCommandComplete - buttonID=" + 
	    gVidReviewUI.buttonPending + ", result=" + 
	    requestResult;
	fastecLogToConsole(gLogAll, my_msg);
	gVidReviewUI.buttonPending = kfipPlayBtnID.FIP_None;
    };

		
    //----------------------------------------------------------
    // sets the display frame to first or last frame 
    //----------------------------------------------------------
    this.displayFirstFrame = function()
    {
	gVidReviewUI.buttonPending = kfipPlayBtnID.FIP_GotoFirst;
	gVidReviewUI.requestFrame(gCameraData.pbCutInFrame);
    };
    this.displayLastFrame = function()
    {
	gVidReviewUI.buttonPending = kfipPlayBtnID.FIP_GotoLast;
	gVidReviewUI.requestFrame(gCameraData.pbCutOutFrame);
    };

    //----------------------------------------------------------
    // Set the display frame to the trigger frame.
    //----------------------------------------------------------
    this.displayTrigFrame = function()
    {
	gVidReviewUI.buttonPending = kfipPlayBtnID.FIP_Trigger;
	gVidReviewUI.requestFrame(gCameraData.actPartTrigFrame);
    };

    //----------------------------------------------------------
    // Called to request a single frame from the camera.
    //----------------------------------------------------------
    this.requestFrame = function(frame_no) {
	fastecLogToConsole(gLogAll, "requestFrame() - frame_no=" + frame_no);

	// Set our current frame ID to the trigger frame.
	gVidReviewUI.currentFrameID = frame_no;

	// Need to make sure the playback engine is stopped.
	var new_control = gCameraData.pbEnabledRegValue & gRegs.GIGE_PBC_INTERVAL_MASK;

        // Store the FCP data using the WSS.
        var fcpFieldKeys = new Array();
        var fcpFieldValues = new Array();

        fcpFieldKeys.push(gRegs.GIGE_REG_PB_START_FRAME);
        fcpFieldValues.push(frame_no);
        fcpFieldKeys.push(gRegs.GIGE_REG_CSI_RVW_FRAME_NUM);
        fcpFieldValues.push(frame_no);
        fcpFieldKeys.push(gRegs.GIGE_REG_PB_CONTROL);
        fcpFieldValues.push(new_control);

        // Invoke the FCP store request
        gFCP_IO.fcpStoreRequest(
            fcpFieldKeys, // the array of field keys to store
            fcpFieldValues, // the array of field values to store
            gCameraData.fcpStoreFieldUpdate, // field update handler
            gVidReviewUI.cbhCommandComplete // CBH after write is complete.
        );
    };

    //----------------------------------------------------------
    // Called to stop playing.
    //----------------------------------------------------------
    this.startPlay = function(forward)
    {
	fastecLogToConsole(gLogAll, "startPlay - forward=" + forward);

        // Store the FCP data using the WSS.
        var fcpFieldKeys = new Array();
        var fcpFieldValues = new Array();
	var new_control = gCameraData.pbEnabledRegValue & gRegs.GIGE_PBC_INTERVAL_MASK;

	// Enable playback.
	new_control |= gRegs.GIGE_PBC_ENABLE;

	// Are we playing in reverse?
	if (forward) {
	    gVidReviewUI.buttonPending = kfipPlayBtnID.FIP_FwdPlay;
	} else {
	    gVidReviewUI.buttonPending = kfipPlayBtnID.FIP_RevPlay;
	    new_control |= gRegs.GIGE_PBC_REVERSE;
	}

	// Send the new control setting to the camera.
        fcpFieldKeys.push(gRegs.GIGE_REG_PB_CONTROL);
        fcpFieldValues.push(new_control);

        // Invoke the FCP store request
        gFCP_IO.fcpStoreRequest(
            fcpFieldKeys, // the array of field keys to store
            fcpFieldValues, // the array of field values to store
            gCameraData.fcpStoreFieldUpdate, // field update handler
            gVidReviewUI.cbhCommandComplete // CBH after write is complete.
        );
    };

    //----------------------------------------------------------
    // Called to step a frame either forward or in reverse.
    //----------------------------------------------------------
    this.step = function(forward) {
	var new_frame = new Number(gVidReviewUI.currentFrameID);
	if (forward) {
	    gVidReviewUI.buttonPending = kfipPlayBtnID.FIP_FwdSkip;
	    new_frame = new_frame + 1;
	    if (new_frame > gCameraData.pbCutOutFrame) {
		new_frame = gCameraData.pbCutInFrame;
	    }
	} else {
	    gVidReviewUI.buttonPending = kfipPlayBtnID.FIP_RevSkip;
	    new_frame = new_frame - 1;
	    if (new_frame < gCameraData.pbCutInFrame) {
		new_frame = gCameraData.pbCutOutFrame;
	    }
	}

	gVidReviewUI.requestFrame(new_frame);
    };

    //----------------------------------------------------------
    // Called to stop playing.
    //----------------------------------------------------------
    this.stopPlay = function() 
    {            
	fastecLogToConsole(gLogAll, "stopPlay");

	gVidReviewUI.buttonPending = kfipPlayBtnID.FIP_Stop;

        // Store the FCP data using the WSS.
        var fcpFieldKeys = new Array();
        var fcpFieldValues = new Array();
	var new_control = gCameraData.pbEnabledRegValue & (gRegs.GIGE_PBC_INTERVAL_MASK | gRegs.GIGE_PBC_SINGLE_ABORT);

	// Send the new control setting to the camera.
        fcpFieldKeys.push(gRegs.GIGE_REG_PB_CONTROL);
        fcpFieldValues.push(new_control);

        // Invoke the FCP store request
        gFCP_IO.fcpStoreRequest(
            fcpFieldKeys, // the array of field keys to store
            fcpFieldValues, // the array of field values to store
            gCameraData.fcpStoreFieldUpdate, // field update handler
            gVidReviewUI.cbhCommandComplete // CBH after write is complete.
        );
    };

    //---------------------------------
    //---------------------------------
    this.updateUI = function() {
	switch (gCameraData.sysCameraState)
	{
            case gRegs.GIGE_CAMERA_STATE_REVIEW:
		gVidReviewUI.show();
		gCameraData.updateReviewRegs();
		break;
	    default:
		gVidReviewUI.hide();
		break;
	}
    };
}
