//------------------------------------------------
// fastec_dialogs.js
//
// Copyright 2010-2018 (c) Fastec Imaging as an unpublished work.
// All Rights Reserved.
//
// The information contained herein is confidential property of
// Fastec Imaging. The use, copying, transfer or disclosure of such
// information is prohibited except by express written agreement
// with Fastec Imaging.
//
// Fastec Imaging Camera Control Web Application
// Dialog Objects --
//------------------------------------------------

//------------------------------------
// The Fastec Message Box OK Cancel
//
// This object is responsible for managing
// a msg box dialog with OK and Cancel.
//------------------------------------
var gMsgBoxOKCancel = new fastecMsgBoxOKCancel();
function fastecMsgBoxOKCancel() {
    try {
        // Used to determine if bindings have taken place.
        this.initializedBindings = false;

        // Holds the callback handler.
        this.callbackMethod = null;

        // Event handler when the user clicks "Cancel".
        this.onCancel = function() {
            if (gMsgBoxOKCancel.callbackMethod != null) {
                gMsgBoxOKCancel.callbackMethod(false);
                gMsgBoxOKCancel.callbackMethod = null;
            }
        };

        // Event handler when the user clicks "OK".
        this.onOK = function() {
            if (gMsgBoxOKCancel.callbackMethod != null) {
                gMsgBoxOKCancel.callbackMethod(true);
                gMsgBoxOKCancel.callbackMethod = null;
            }
        };

        //---------------------------
        // shows modal dialog with ok/cancel
        //---------------------------
        this.showMessageBox = function(
            inTitle,
            inMsgBody,
            callMeBackWithResult
        ) {
            // Setup the callback method.
            this.callbackMethod = callMeBackWithResult;

            // Have we done our bindings?
            if (!this.initializedBindings) {
                // Bive our object a way to bind events to it
                this.myObjData = { fastecMsgBoxOKCancel: this };

                $("#dialogMessageBoxOK").bind(
                    "click",
                    this.myObjData,
                    this.onOK
                );
                $("#dialogMessageBoxCancel").bind(
                    "click",
                    this.myObjData,
                    this.onCancel
                );
                this.initializedBindings = true;
            }

            // Set the title
            $("#messageBoxTitle").html(inTitle);

            // Setup the text.
            $("#messageBoxTextID").html(inMsgBody);

            // show the message box
            $("#dialogMessageBox").popup("open");
        };
    } catch (err) {
        var theDetails = "caught: fastecMsgBoxOKCancel-" + err.description;
        gFaultHandler.logError(theDetails);
    }
}

//------------------------------------
// The Fastec Wait Box Dialog Window
//
// This object is responsible for managing
// a wait box dialog.
//------------------------------------
var gWaitMsgBox = new fastecWaitMsgBox();
function fastecWaitMsgBox() {
    try {
        this.callbackMethod = null;
        this.onClose = function() {
            if (gWaitMsgBox.callbackMethod != null) {
                gWaitMsgBox.callbackMethod(false);
                gWaitMsgBox.callbackMethod = null;
            }

            $("#waitOnCloseMsgID").html("");
            $("#waitBoxImgID").hide();
            $("#dialogWaitBox").popup("close");
        };

        //---------------------------
        // shows modal dialog with ok/cancel
        //---------------------------
        this.showWaitBox = function(
            inTitle,
            inMsgBody,
            inOnCloseWarnMsg,
            callMeBackWithResult
        ) {
            this.callbackMethod = callMeBackWithResult;

            // Setup the text in the message box.
            $("#waitBoxTitle").html(inTitle);
            $("#waitOnCloseMsgID").html(inMsgBody);

            // Enable the waiting animated gif.
            $("#waitBoxImgID").show();

            // Show the message box
            $("#dialogWaitBox").popup("open");
        };
    } catch (err) {
        var theDetails = "caught: fastecWaitMsgBox-" + err.description;
        gFaultHandler.logError(theDetails);
    }
}

//------------------------------------
// The Fastec Dearm Dialog Window
//
// This object is responsible for managing
// a popup dialog allowing for dearm state
// notification and direction
//------------------------------------
var gDeArmDialog = new fastecDeArmDialog();
function fastecDeArmDialog() {
    var dearm_str = {
        dearmdlg_Title: "!! Stop Capture Warning !!",
        dearmdlg_Info:
            "If you stop recording now, you may use the checkbox to choose whether to retain or disgard the current session.<br>",
        dearmdlg_ActionInfo:
            "<br>Select Continue to keep recording or Stop if you really want to stop recording now.<br>",
        dearmdlg_AppendModeInfo:
            "Select Stop Capture to exit Video Capture so you can Review or Save the recording.<br><br>Or Select Continue Capture to remain in Video Capture where you can use the Trigger button to run and pause (ROC) or to capture a burst (BROC).<br>",
        dearmdlg_AppendModeActionInfo: " ",

        dearmdlg_Retain: "Retain Current Session",
        dearmdlg_Stop: "Stop Capture",
        dearmdlg_Continue: "Continue Capture"
    };

    try {
        // who to callback when dialog closes -
        // callback signature takes in stop button boolean and retain check state
        this.callbackMethod = null;

        // used to determine if bindings have taken place.
        this.initializedBindings = false;

        // what is the state of retaining the capture setting
        // it's default should be true (checked).
        this.retainCapture = true;

        // will keep track of how the dialog was terminated
        this.stopButtonWasHit = false;

        //--------------------------------------
        // stop button click handler
        //--------------------------------------
        this.onStopClick = function(e) {
            if (e.data.fiDeArmDialogOBJ != undefined) {
                var me = e.data.fiDeArmDialogOBJ;
                me.stopButtonWasHit = true;
            }
        };

        //--------------------------------------
        // continue button click handler
        //--------------------------------------
        this.onContinueClick = function(e) {
            if (e.data.fiDeArmDialogOBJ != undefined) {
                var me = e.data.fiDeArmDialogOBJ;
                me.stopButtonWasHit = false;
            }
        };

        //-----------------------------------
        //-----------------------------------
        this.onRetainChange = function(e) {
            if (e.data.fiDeArmDialogOBJ != undefined) {
                var me = e.data.fiDeArmDialogOBJ;

                // Toggle the state.
                me.retainCapture = !me.retainCapture;

                // There's probably a better way...
                /*
		if (me.retainCapture)
		{
		    $('#deArmRetainCapLabel').removeClass("ui-checkbox-off");
		    $('#deArmRetainCapLabel').addClass("ui-checkbox-on");
		}
		else
		{
		    $('#deArmRetainCapLabel').removeClass("ui-checkbox-on");
		    $('#deArmRetainCapLabel').addClass("ui-checkbox-off");
		}
		*/
            }
        };

        //-----------------------------------
        //-----------------------------------
        this.initBindings = function() {
            // give our object a way to bind events to it
            this.myObjData = { fiDeArmDialogOBJ: this };

            // bind handlers for the OK and Cancel buttons.
            $("#deArmStopBTN").bind("click", this.myObjData, this.onStopClick);
            $("#deArmContinueBTN").bind(
                "click",
                this.myObjData,
                this.onContinueClick
            );

            // create a check box for retain capture state
            // install the retain check box string
            $("#deArmRetainCapLabel").html(dearm_str.dearmdlg_Retain);
            $("#deArmRetainCap").bind(
                "click",
                this.myObjData,
                this.onRetainChange
            );

            /*
            // IF this camera is in Long Record Append Mode Do Not show the retain information but
            // go ahead and set it anyway
            if (gCameraData.isCameraInLRAppendMode())
            {
                this.retainCapCheckBox.setCheck(true);
                $('#deArmRetainDivID').hide();
            }
            else    // normal operations will show this retain check box.
	    */
            {
                $("#deArmRetainDivID").show();
            }

            this.initializedBindings = true; // we are initialized
        };

        //---------------------------
        // shows modal dialog
        //---------------------------
        this.showDeArmDialog = function(inCallbackFunc) {
            // install the callback function
            this.callbackMethod = inCallbackFunc;

            // how did dialog terminate it will be either stop button or close/continue
            this.stopButtonWasHit = false;

            // default retain capture to checked (true)
            this.retainCapture = true;

            // if haven't initialized bindings do so
            if (!this.initializedBindings) this.initBindings();

            // setup the dialog options
            var dialogOpts = {
                afterclose: function(ev, ui) {
                    if (gDeArmDialog.callbackMethod != null) {
                        gDeArmDialog.callbackMethod(
                            gDeArmDialog.stopButtonWasHit,
                            gDeArmDialog.retainCapture
                        );
                        gDeArmDialog.callbackMethod = null;
                    }
                }
            };

            // create the dialog element
            $("#dialogDeArm").popup(dialogOpts);

            // Set the title
            $("#dearmTitle").html(dearm_str.dearmdlg_Title);

            // Install the information string.
            $("#dearmInfo").html(dearm_str.dearmdlg_Info);

            // Install the action information string.
            $("#dearmActionInfo").html(dearm_str.dearmdlg_ActionInfo);

            // Set text for the Stop and Continue buttons.
            $("#deArmStopBTN").html(dearm_str.dearmdlg_Stop);
            $("#deArmContinueBTN").html(dearm_str.dearmdlg_Continue);

            // Set/clear the checkbox as appropriate. Note that we default to setting this
            // true (see above).
            $("#deArmRetainCap")
                .prop("checked", gDeArmDialog.retainCapture)
                .checkboxradio("refresh");

            /*
            // circumvent display on the dialog if dearming from a Long Record Append Mode
            if (gCameraData.isCameraInLRAppendMode())
            {
                $('#dearmInfo').html(dearm_str.dearmdlg_AppendModeInfo);
                $('#dearmActionInfo').html(dearm_str.dearmdlg_AppendModeActionInfo);
            }
	    */

            // show the message box
            $("#dialogDeArm").popup("open");
        };
    } catch (err) {
        var theDetails = "caught: dialogDeArm-" + err.description;
        gFaultHandler.logError(theDetails);
    }
}
