//------------------------------------------------
// main.js
//
// Copyright 2018 (c) Fastec Imaging as an unpublished work.
// All Rights Reserved.
//
// The information contained herein is confidential property of
// Fastec Imaging. The use, copying, transfer or disclosure of such
// information is prohibited except by express written agreement
// with Fastec Imaging.
//
// Fastec Imaging Camera Control Web Application
// Main entry for the JQuery support functionality
//------------------------------------------------

//----------------
// Global Objects
//----------------

var gFastecAppMgr = null; // The global state management object.

var gArmTrigPctCtrl = null; // The global percent control for arming and triggering.
var gCameraData = null; // The global camera data object.
var gFCP_IO = null; // The FCP IO object used for camera data exchanges.
var gFCP_StateListener = null; // The FCP State Listener object used for handling asynchronous notifications.
var gFCP_StateIO = null; // The FCP IO object used for retrieving state info.
var gPanelMgr = null; // The overall panel manager object.
var gToolbarMgr = null; // The global toolbar management object.
var gVidReviewUI = null; // The global review UI management object.

//------------------------------------
// The Fastec Application Manager Object
//
// This object is responsible for managing
// the state of this web application.
//------------------------------------
function fastecAppMgr() {
    fastecLogToConsole(gLogTrace, "fastecAppMgr()");

    appmgr_str = {
        app_preset: "Preconfiguring System",
        app_initializing: "Loading System",
        app_uninitialized: "Waiting to Initialize",
        app_sysLoaded: "System Loaded",
        app_sysLoadFail: "System Load Failure",
        app_resetToFactory: "Reset To Factory",
        app_poweringDown: "Power Down Camera",
        app_blackRefCalib: "Black Ref Calibration",
        app_fmtMediaOp: "Formatting Media",
        camIdle: "Idle",
        camLive: "Live",
        camArmed: "Armed",
        camDearmed: "DeArming",
        camTrig: "Triggered",
        camReview: "Review",
        camSnapshot: "Snapshot",
        camAutosave: "Autosave",
        errNetworkDown: "Communication with Camera over Network has been Interrupted",
        errSaveFail: "(Camera Store Failure)",
        errSysLoad: "System Load Failure",
        initialLoadFail: "<br>The Camera System was Unable to Load.<br><br>Use Browser Reload to Try Again.",
        lostNetwork: "Network Connection Failure",
        networkGoneTitle: "Lost Connection to Camera",
        networkGoneInfo:
            "The Network Connection to Camera is Lost.  Please Refresh the Application once Camera is Rebooted and Network Connection is Available.",
        OkToLoseCaptureTitle: "Arm Warning",
        OkToLoseCaptureBody:
            "Arming the camera will result in the unrecoverable loss of any unsaved video.<br><br>Are you sure you want to continue with the Arm request?",
        OkToStopCaptureTitle: "Terminate Capture Warning",
        OkToStopCaptureBody:
            "De-Arming the camera will result in the unrecoverable loss of any unsaved video.<br><br>Are you sure you want to continue with the Live request?",
        waitingToArm: "Waiting for ARM",
        waitingToDeArm: "Waiting for DE-ARM",
        waitingToTrigger: "Waiting for Trigger",
        waitingForLive: "Waiting for Live View",
        waitingForReview: "Waiting for Trigger Complete",
        waitingForAutoSavingComplete: "Waiting for Autosave Writes",
        waitingForInitialLoad: "Loading Camera System<br>Please Wait.",
        wss_fcpInfoDisconnect: "FCP WSS Was Disconnected",
        wss_fcpInfoSysLoadErr: "<br><br>*Happened Before System Load Completed"
    };

    // State of this Application
    var fastecAppStates = {
        fastecUnInit: -1, // the application is not initialized
        fastecAppPreset: 0, // the application is presetting pulling info from php for configuration
        fastecAppInit: 1, // the application is initializing
        fastecSysLoadFail: 2, // camera system information unable to load
        fastecSysLoaded: 3, // camera system information is loaded
        fastecNetworkGone: 4, // camera has lost network connection
        // DON'T CHANGE ORDER BELOW!!!!!
        fastecCamLive: 5, // camera is in live mode so are we
        fastecCamArmed: 6, // camera is in armed mode so are we
        fastecCamDeArmed: 7, // camera is in de-armed mode so are we
        fastecWaitingToArm: 8, // camera system has been asked to arm itself
        fastecWaitingToDeArm: 9, // camera system has been asked to de-arm itself
        fastecWaitingForLive: 10, // camera system has been asked to go back to Live view
        fastecWaitingToTrigger: 11, // camera system has been asked to trigger itself
        fastecCamTriggered: 12, // camera is in trigger mode so are we
        fastecWaitingToReview: 13, // camera system has been asked to go to review mode
        fastecCamReview: 14, // camera is in review mode we are in holding pattern
        fastecCamAutosave: 15,
        fastecCamOther: 16, // camera is in an idle snapshot we are in holding pattern
        fastecResetToFactory: 17, // camera is in reset to factory mode (camera state==idle)
        fastecSentResetToFactory: 18, // camera has been sent the reset factory command
        fastecUpdateCamera: 19, // camera is in update mode (camera state==idle)
        fastecPowerDown: 20, // camera is in powerdown mode (camera state==idle)
        fastecUserConfig: 21, // camera is processing save/load user config (camera state==idle)
        fastecBlkRefCalib: 22, // camera is processing black reference frame (camera state goes idle)
        fastecFmtMedia: 23, // camera is processing format media (camera state goes idle)
        fastecWaitingForAutoSaving: 24 // need case when user wants to go live but is still waiting for autosaving to be complete
    };

    this.theState = fastecAppStates.fastecUnInit; // Default to uninitialized state.
    this.systemHasBeenLoaded = false; // Set true once a full system load has succeeded.

    // These variables will manage camera state updates
    this.pendingCameraStateRequest = false; // waiting for camera state request to be serviced
    this.pendingCameraStateChange = false; // waiting for camera state change to return

    //--------------------------------------
    // Callback Handler for when the user is
    // dearming the camera.
    //--------------------------------------
    this.cbhDeArmDialog = function(dearmOK, retainCapture) {
        // If we aren't de-arming, we can leave.
        if (!dearmOK) {
	    return;
	}

        // If the user wants to retain the capture data we go into a dearm state.
        if (!retainCapture) {
            // If autosaving is in progress or enabled need to disable then go live.
            if (gCameraData.autoSaveRegEnabled || gCameraData.autoSaveSaving) {
                gCameraData.disableAutoSave(gFastecAppMgr.onOKToGoLive);
	    } else {
		gFastecAppMgr.onOKToGoLive(true); // go live at this point...
	    }
        } else {
            // Put camera server into a dearm state to let the camera server cleanup
            // and finish the capture. It should automatically drop into the proper
            // autosave or review when complete.
            gFastecAppMgr.setCameraStateDeArmed();
        }
    };

    //--------------------------------------
    // Callback Handler for when the camera state
    // change request was completed.
    //--------------------------------------
    this.cbhSetCameraState = function(changedOK) {
        gFastecAppMgr.pendingCameraStateChange = false;

        if (!changedOK) {
	    fastecLogToConsole(gLogAll, "cbhSetCameraState()" + " changedOK=" + changedOK);
	}
    };

    //-------------------------------------------
    // Callback Handler for Notifications about
    // Liveview Connections
    //-------------------------------------------
    this.cbhFIWSS_LVConnected = function() {
        fastecLogToConsole(gLogTrace, "cbhFIWSS_LVConnected()");

        // Force a click to take place...
        // which should start the live view if
        // it is ok to do so...
        gFastecAppMgr.onLiveViewToggleTB();
    };

    //-------------------------------------------
    // Callback Handler for Notifications about
    // Liveview Disconnections
    //-------------------------------------------
    this.cbhFIWSS_LVDisConnected = function() {
        fastecLogToConsole(gLogTrace, "cbhFIWSS_LVDisConnected()");

        // must refresh in order to achieve re-attempted connection
        // set (active,play) to (false,true)
        gToolbarMgr.setLiveViewToggleTBState(false, true);
    };

    //---------------------------
    // Application State Change
    //---------------------------
    this.changeAppState = function(theNewState) {
        fastecLogToConsole(gLogTrace, "changeAppState(" + theNewState + ")");

        switch (theNewState) {
            case fastecAppStates.fastecAppInit:
                gFastecAppMgr.vfInfoBoxShow(appmgr_str.waitingForInitialLoad);
                gFastecAppMgr.theState = theNewState;
                break;
            case fastecAppStates.fastecSysLoadFail:
                gFastecAppMgr.vfInfoBoxShow(appmgr_str.initialLoadFail);
                gFastecAppMgr.systemHasBeenLoaded = false;

                // Update the UI.
                gFastecAppMgr.updateCameraStatus(appmgr_str.errSysLoad);

                // Provide the user with dialog of errors returned FCP system load.
                gFastecAppMgr.handleFailedSystemLoad(gFCP_IO, null);

                // This will be the end of our application.
                gFastecAppMgr.theState = theNewState;
                break;
            case fastecAppStates.fastecSysLoaded:
                gFastecAppMgr.systemHasBeenLoaded = true;

                // Load up the camera information data.
                $("#bannerCamera").html(gCameraData.cameraName);

                // Remove any messages we are going live.
                gFastecAppMgr.vfInfoBoxHide();

                // Set state.
                gFastecAppMgr.theState = theNewState;

                // Force initial AR setting
                videoSyncWithAR(true);

                // Make sure control panel availability matches camera state.
                gFastecAppMgr.updateUIForCameraState(gCameraData.sysCameraState);
                gFastecAppMgr.updateRecSettings();
                gFastecAppMgr.updateCameraInfo();

                // Initialize the controls on the panel.
                gPanelMgr = new fastecPanelMgr();
                gPanelMgr.initPanel();
                break;
            case fastecAppStates.fastecCamLive:
                gFastecAppMgr.theState = theNewState;
                gFastecAppMgr.updateCameraStatus(appmgr_str.camLive);
                gFastecAppMgr.vfInfoBoxHide();
                break;
            case fastecAppStates.fastecCamArmed:
                gFastecAppMgr.theState = theNewState;
                gFastecAppMgr.updateCameraStatus(appmgr_str.camArmed);
                gFastecAppMgr.vfInfoBoxHide();

                var triggerInfo = gCameraData.getTriggerInfo();
                var myMsg = "ARM TFrame[" + triggerInfo.tFrame + "] sys_cur_frame_no[" + gCameraData.sys_cur_frame_no + "]";
                fastecLogToConsole(gLogTriggerInfo, myMsg);

                gArmTrigPctCtrl.setMinMaxCur(0, triggerInfo.maxFrames, gCameraData.maxArmFrameNo);
                gArmTrigPctCtrl.setGaugeColor("Green");
                gArmTrigPctCtrl.setTFrame(triggerInfo.tFrame);
                gArmTrigPctCtrl.flashGauge(true);
                break;
            case fastecAppStates.fastecCamTriggered:
                gFastecAppMgr.theState = theNewState;
                gFastecAppMgr.updateCameraStatus(appmgr_str.camTrig);
                gFastecAppMgr.vfInfoBoxHide();

                // Set the trigger frame
                var triggerInfo = gCameraData.getTriggerInfo();
                var myMsg = "TRIG maxArmFrameNo[" + gCameraData.maxArmFrameNo + "] maxFrames[" + triggerInfo.maxFrames + "]";
                fastecLogToConsole(gLogTriggerInfo, myMsg);

                gArmTrigPctCtrl.setGaugeColor("Red");
                gArmTrigPctCtrl.setMinMaxCur(0, triggerInfo.maxFrames, gCameraData.maxArmFrameNo);
                gArmTrigPctCtrl.setTFrame(gCameraData.maxArmFrameNo);
                gArmTrigPctCtrl.flashGauge(false);
                break;
            case fastecAppStates.fastecCamReview:
                gFastecAppMgr.theState = theNewState;
                gFastecAppMgr.updateCameraStatus(appmgr_str.camReview);
                gFastecAppMgr.vfInfoBoxHide();
                break;
	    case fastecAppStates.fastecWaitingToTrigger:
                gFastecAppMgr.updateCameraStatus(appmgr_str.waitingToTrigger);
                gFastecAppMgr.theState = theNewState;
                gFastecAppMgr.vfInfoBoxHide();

                gArmTrigPctCtrl.setGaugeColor("Red");
		break;

            case fastecAppStates.fastecNetworkGone:
                gFastecAppMgr.updateCameraStatus(appmgr_str.lostNetwork);
                gFastecAppMgr.theState = theNewState;
                gFastecAppMgr.vfInfoBoxShow(appmgr_str.errNetworkDown);

                gMsgBoxOKCancel.showMessageBox(appmgr_str.networkGoneTitle, appmgr_str.networkGoneInfo);
                break;
        }

        gFastecAppMgr.syncToolbarButtons();
    };

    //--------------------------------------
    // Central point for making state changes to camera.
    //--------------------------------------
    this.changeCameraState = function(theNewState) {
        if (!gFastecAppMgr.pendingCameraStateChange) {
            fastecLogTimeToConsole(gLogStatePolling, "DRIVING STATE CHANGE TO[" + theNewState + "]");
            gFastecAppMgr.pendingCameraStateChange = true;

            gCameraData.setCameraState(theNewState, gFastecAppMgr.cbhSetCameraState);
        }
    };

    //-------------------------------------------
    // FCP IO Object Disconnected
    //-------------------------------------------
    this.fcpIODisconnected = function() {
        // Something wrong lost FCP IO connection
        fastecLogToConsole(gLogTrace, "fcpIODisconnected()");

        // Something wrong - lost FCP IO connection
        gFastecAppMgr.changeAppState(fastecAppStates.fastecSysLoadFail);
    };

    //-------------------------------------------
    // FCP IO Object Connected [Chaining Order Step 2]
    //-------------------------------------------
    this.fcpIOConnected = function() {
        fastecLogToConsole(gLogTrace, "fcpIOConnected()");

        try {
            // Have main FCP IO connection try to create FCP State IO object
            gFCP_StateIO = new fiFCP_IO(0, gFastecAppMgr.fcpStateIOConnected, gFastecAppMgr.fcpStateIODisconnected);
            gFCP_StateIO.prepFCPIO();
        } catch (err) {
            var theDetails = "caught: fcpIOConnected -" + err.description;
            gFaultHandler.logError(theDetails);
        }
    };

    //-------------------------------------------
    // FCP Listener Object Disconnected Terminate chain
    //-------------------------------------------
    this.fcpListenerDisconnected = function() {
        fastecLogToConsole(gLogTrace, "fcpListenerDisconnected()");

        // Something wrong - lost FCP Listener connection.
        gFastecAppMgr.changeAppState(fastecAppStates.fastecSysLoadFail);
    };

    //-------------------------------------------
    // FCP Listener Object Connected Step 4
    //
    // All of the FCP objects are created and now
    // this will chain to the main system load function
    //-------------------------------------------
    this.fcpListenerConnected = function() {
        fastecLogToConsole(gLogTrace, "fcpListenerConnected()");

        try {
            // Have the State Listener now proceed Loading System Data.
            gCameraData.loadSystemData(gFastecAppMgr.initAfterCameraLoad);
        } catch (err) {
            var theDetails = "caught: fcpListenerConnected -" + err.description;
            gFaultHandler.logError(theDetails);
        }
    };

    //-------------------------------------------
    // FCP State IO Object Disconnected Terminate chain
    //-------------------------------------------
    this.fcpStateIODisconnected = function() {
        fastecLogToConsole(gLogTrace, "fcpStateIODisconnected()");

        // Provide the user with dialog of errors returned FCP system load.
        gFastecAppMgr.handleFCPDisconnection(gFCP_StateIO, null);

        // Something wrong lost FCP IO connection.
        // If system is not loaded set system load fail.
        if (!gFastecAppMgr.systemHasBeenLoaded) gFastecAppMgr.changeAppState(fastecAppStates.fastecSysLoadFail);
        else gFastecAppMgr.onNetworkLostConn();
    };

    //-------------------------------------------
    // FCP State IO Object Connected [Chaining Order Step 3]
    //-------------------------------------------
    this.fcpStateIOConnected = function() {
        fastecLogToConsole(gLogTrace, "fcpStateIOConnected()");

        // Create the State Listener object
        gFCP_StateListener = new fiFCP_IO(1, gFastecAppMgr.fcpListenerConnected, gFastecAppMgr.fcpListenerDisconnected);
        gFCP_StateListener.prepFCPIO();
    };

    //--------------------------------------
    // Handle failed system load
    //--------------------------------------
    this.handleFailedSystemLoad = function(theFCPIO, callOnClose) {
        var whyFailed = theFCPIO.getTheStatusMsg();
        gMsgBoxOKCancel.showMessageBox("System Load Failure", whyFailed, callOnClose);
    };

    //--------------------------------------
    // Handle disconnected FCP
    //--------------------------------------
    this.handleFCPDisconnection = function(theFCPIO, callOnClose) {
        var whyFailed = theFCPIO.getTheStatusMsg();

        // Note the system load status.
        if (!gFastecAppMgr.systemHasBeenLoaded) {
	    whyFailed = whyFailed + appmgr_str.wss_fcpInfoSysLoadErr;
	}

        gMsgBoxOKCancel.showMessageBox(appmgr_str.wss_fcpInfoDisconnect, whyFailed, callOnClose);
    };

    //--------------------------------------
    // Handle failed FCP Store
    //--------------------------------------
    this.handleFailedFCPStore = function(theFCPIO, callOnClose) {
        var whyFailed = theFCPIO.getTheStatusMsg();
        gMsgBoxOKCancel.showMessageBox(appmgr_str.errSaveFail, whyFailed, callOnClose);
    };

    //-------------------------------------------
    // Handles State after Initial Camera Load
    //-------------------------------------------
    this.initAfterCameraLoad = function(loadedOK) {
        fastecLogToConsole(gLogTrace, "initAfterCameraLoad() loadedOK:" + loadedOK);

        // Reset global state in the camera -
        // this should be better handled but using global object right now
        gCameraData.loadingSysInfo = false;
        gCameraData.loadedCameraData = loadedOK;

        // If the data was loaded from camera proceed with the UI initialization.
        if (loadedOK) {
            // Initialize the websocket server listener connection.
            g_fiwss_listener = new fiwss_listener();
            g_fiwss_listener.create_wss_socket();

            // Initialize the web socket server liveview connection.
            g_fiwss_liveview = new fiwss_liveview(gFastecAppMgr.cbhFIWSS_LVConnected, gFastecAppMgr.cbhFIWSS_LVDisConnected);
            g_fiwss_liveview.create_wss_socket();

	    // Create our playback connection.
	    g_fiwss_playback = new fiwss_playback();
	    g_fiwss_playback.create_wss_socket();

	    // Connect with the WSS playback object so we can update
	    // our display when an image is received.
	    if (g_fiwss_playback == null) {
		fastecLogToConsole(gLogAll, "fastecReviewMgr(): g_fiwss_playback is NULL!");
		return;
	    }
	    g_fiwss_playback.cbhCallMeWhenPBFrameArrives = gVidReviewUI.cbhFrameReceived;


            // We can now switch to the loaded state.
            gFastecAppMgr.changeAppState(fastecAppStates.fastecSysLoaded);

            // Here is where we want to setup the notification for
            // the state information.
            gCameraData.setupCameraNotifications(gFastecAppMgr.onCameraInfoUpdate);
        } else {
            gFastecAppMgr.changeAppState(fastecAppStates.fastecSysLoadFail);
        }
    };

    //---------------------------
    // Application initialization
    //---------------------------
    this.initFastecWebApp = function() {
        fastecLogToConsole(gLogTrace, "initFastecWebApp()");

        try {
            // Create our camera data object.
            gCameraData = new cameraData();

            // Create our toolbar management object.
            gToolbarMgr = new fastecToolbarMgr();
            gToolbarMgr.initFastecToolbar();

            // Create our ARM time progress bar.
            gArmTrigPctCtrl = new fastecArmTrigCtl();
	    gArmTrigPctCtrl.initCtl();

	    // Create the video review UI management object.
	    gVidReviewUI = new fastecReviewMgr();
	    gVidReviewUI.hide();

            // Setup the WSS connections...
            // Initialize the global FCP WSS IO objects
            gFastecAppMgr.initFCPObjects();
        } catch (err) {
            var theDetails = "caught: initFastecWebApp -" + err.description;
            gFaultHandler.logError(theDetails);
        }
    };

    //-------------------------------------------
    // Initialize the FCP Objects [Chaining Order Step 1]
    //-------------------------------------------
    this.initFCPObjects = function() {
        fastecLogToConsole(gLogTrace, "initFCPObjects()");

        try {
            // Use chaining and callbacks to cascade WSS object creation.
            // Pass a '0' because this is a load/store IO object (not a listener).
            gFCP_IO = new fiFCP_IO(0, gFastecAppMgr.fcpIOConnected, gFastecAppMgr.fcpIODisconnected);
            gFCP_IO.prepFCPIO();
        } catch (err) {
            var theDetails = "caught: initFCPObjects -" + err.description;
            gFaultHandler.logError(theDetails);
        }
    };

    //---------------------------
    // isInOrWaitingForCapture will
    // test the state to see if camera is either
    // waiting for a capture or in capture (armed or triggered)
    //---------------------------
    this.isInOrWaitingForCapture = function() {
        if (
            gFastecAppMgr.theState == fastecAppStates.fastecWaitingToTrigger ||
            gFastecAppMgr.theState == fastecAppStates.fastecCamTriggered ||
            gFastecAppMgr.theState == fastecAppStates.fastecCamArmed ||
            gFastecAppMgr.theState == fastecAppStates.fastecWaitingToArm
        ) {
            return true;
        }

        return false;
    };

    //---------------------------
    // isInDeArm will
    // test the state to see if camera is
    // in a dearming state
    //---------------------------
    this.isInDeArm = function() {
        if (
            gFastecAppMgr.theState == fastecAppStates.fastecWaitingToDeArm ||
            gFastecAppMgr.theState == fastecAppStates.fastecCamDeArmed
        ) {
            return true;
        }

        return false;
    };

    //---------------------------
    // ARM Button Click Handler
    //---------------------------
    this.onArmTBClick = function() {
        // Set the arm/trig buttons based on app state.
        if (gFastecAppMgr.isInOrWaitingForCapture()) {
            // Invoke the dearm dialog to determine what to do.
            gDeArmDialog.showDeArmDialog(gFastecAppMgr.cbhDeArmDialog);
        } else if (!gFastecAppMgr.isInDeArm()) {
            // Before allowing Arm make sure there isn't any captured data.
            if (!gCameraData.isCameraInLRAppendMode() && (gCameraData.hasCaptureData() || gCameraData.autoSaveSaving)) {
                gMsgBoxOKCancel.showMessageBox(
                    appmgr_str.OkToLoseCaptureTitle,
                    appmgr_str.OkToLoseCaptureBody,
                    gFastecAppMgr.onOKToArm
                );
            } else {
                gFastecAppMgr.setCameraStateArmed();
            }
        }
    };

    //---------------------------
    // Camera State Reloaded - callback from state reload
    //---------------------------
    this.onCameraInfoUpdate = function(loadedOK) {
        var myMsg =
            "onCameraInfoUpdate LOADOK[" +
            loadedOK +
            "] STATE[" +
            gCameraData.sysCameraState +
            "] ASEnabled,Saving [" +
            gCameraData.autoSaveRegEnabled +
            "," +
            gCameraData.autoSaveSaving +
            "] SysCurFrame[" +
            gCameraData.sys_cur_frame_no +
            "] appState[" +
            gFastecAppMgr.theState +
            "]";
        if (!loadedOK) {
	    fastecLogTimeToConsole(gLogStatePolling, myMsg);
	}

        // Are we in the middle of a reset to factory operation?
        if (gFastecAppMgr.theState === fastecAppStates.fastecResetToFactory) {
            var myMsg =
                "RESET FACTORY CHECK Camera State[" + gCameraData.sysCameraState + "] AppState[" + gFastecAppMgr.theState + "]";
            fastecLogTimeToConsole(gLogStatePolling, myMsg);

            // If the idle state is achieved need to process the request.
            if (gCameraData.sysCameraState == gRegs.GIGE_CAMERA_STATE_IDLE) {
                // Set the app state as a waiting for reset to factory command
                gFastecAppMgr.theState = fastecAppStates.fastecSentResetToFactory;
            }
            return;
        }

        // Are we in the middle of a black level calibration?
        if (gFastecAppMgr.theState == fastecAppStates.fastecBlkRefCalib) {
            var myMsg =
                "BLACK REF STATE CHECK[" +
                gCameraData.sysCameraState +
                "] #Req[" +
                gFastecAppMgr.nbrBlkLevelCalibStateReqs +
                "] flag[" +
                gFastecAppMgr.inBlkLevelCalib +
                "]";
            fastecLogTimeToConsole(gLogStatePolling, myMsg);
        }

        // Are we in the middle of a format media operation?
        if (gFastecAppMgr.theState == fastecAppStates.fastecFmtMedia) {
            var myMsg =
                "FMT MEDIA CHECK[" +
                gCameraData.sysCameraState +
                "] #Req[" +
                gFastecAppMgr.nbrFmtMediaStateReqs +
                "] flag[" +
                gFastecAppMgr.inFmtMedia +
                "]";
            fastecLogTimeToConsole(gLogStatePolling, myMsg);
        }

        // If simply responding to state change without special cases above
        if (loadedOK) {
            fastecLogToConsole(
                gLogStatePolling,
                "onCameraInfoUpdate(LOADEDOK)" +
                    " curFrame=" +
                    gCameraData.sys_cur_frame_no +
                    " maxFrame=" +
                    gCameraData.maxArmFrameNo +
                    " camera_state=" +
                    gCameraData.sysCameraState +
                    " part_state=" +
                    gCameraData.actPartState +
                    " appstate=" +
                    gFastecAppMgr.theState +
                    " asE=" +
                    gCameraData.autoSaveRegEnabled +
                    " asSv=" +
                    gCameraData.autoSaveSaving +
                    " asO=" +
                    gCameraData.autoSaveOlapEnabled +
                    " asC=" +
                    gCameraData.autoSaveCount +
                    " asT=" +
                    gCameraData.autoSaveTotal
            );

            gFastecAppMgr.updateUIForCameraState(gCameraData.sysCameraState);
            gFastecAppMgr.updateRecSettings();
            gFastecAppMgr.refreshCameraStatus();
        }
    };

    //---------------------------
    // LIVE Button Click Handler
    //---------------------------
    this.onLiveTBClick = function() {
        // If in middle of pre/post trigger before partition complete...
        if (
            gFastecAppMgr.theState == fastecAppStates.fastecWaitingToTrigger ||
            gFastecAppMgr.theState == fastecAppStates.fastecCamTriggered ||
            gFastecAppMgr.theState == fastecAppStates.fastecCamArmed ||
            gFastecAppMgr.theState == fastecAppStates.fastecWaitingToArm
        ) {
            // Make sure the user is OK with this.
            gMsgBoxOKCancel.showMessageBox(
                appmgr_str.OkToStopCaptureTitle,
                appmgr_str.OkToStopCaptureBody,
                gFastecAppMgr.onOKToGoLive
            );
        } else {
            // Should be ok to go live.
            gFastecAppMgr.onOKToGoLive(true);
        }
    };

    //---------------------------
    // LIVE VIEW stream toggle Toolbar Click Handler
    //---------------------------
    this.onLiveViewToggleTB = function() {
        fastecLogToConsole(gLogTrace, "onLiveViewToggleTB()");

        var liveViewTBActive = false;

        // If the connection was lost then all we can really do here is
        // to reattempt connection to prepare for liveview streaming.
        if (!g_fiwss_liveview.is_connected()) {
            // Are we not connected yet or have we lost connection?
            if (g_fiwss_liveview.lost_connection) {
                // An onclick when a connection has been lost should
                // attempt to reconnect to the fiwss server.
                g_fiwss_liveview.reconnect();

                // Set (active,play) to (true,true)
                gToolbarMgr.setLiveViewToggleTBState(true, true);
            } else {
                // Just keep play active right now and do nothing.
                // Set (active,play) to (true,true)
                gToolbarMgr.setLiveViewToggleTBState(true, true);
            }
        } else {
            // We are connected -- is the protocol streaming or not right
            // now which determines what we will do on click.
            if (g_fiwss_liveview.is_streaming()) {
                // Turn off streaming and reset the button to false
                fiLiveViewOffClicked();

                // Set (active,play) to (true,true)
                gToolbarMgr.setLiveViewToggleTBState(true, true);
            } else {
                // If we can stream then allow it to happen.
                if (
                    gFastecAppMgr.theState == fastecAppStates.fastecCamLive ||
                    gFastecAppMgr.theState == fastecAppStates.fastecCamArmed
                ) {
                    // Turn on streaming and reset the button to false
                    fiLiveViewOnClicked();

                    // Set (active,play) to (true,false)
                    gToolbarMgr.setLiveViewToggleTBState(true, false);
                } else {
                    // Not in a live view state make inactive.
                    // Set (active,play) to (false,true)
                    gToolbarMgr.setLiveViewToggleTBState(false, true);
                }
            }
        }
    };

    //---------------------------
    // Lost Network Connection
    //---------------------------
    this.onNetworkLostConn = function() {
        gFastecAppMgr.changeAppState(fastecAppStates.fastecNetworkGone);

        // Hide everything and make nothing available.
        $("#videoAreaID").hide();
        $("#bannerCamera").hide();
        $("#videoCaptureSectionID").hide();
    };

    //--------------------------------------
    // callback from user saying its ok to
    // get rid of the data in partition
    //--------------------------------------
    this.onOKToArm = function(isItOK) {
        if (isItOK) {
            // If autosaving must disable the register first then arm
            if (gCameraData.autoSaveSaving) gCameraData.disableAutoSave(gFastecAppMgr.onOKToArmAfterDisableAutosave);
            else gFastecAppMgr.onOKToArmAfterDisableAutosave(true);
        }
    };

    //--------------------------------------
    // callback from user saying its ok to
    // get rid of the data in partition
    //--------------------------------------
    this.onOKToArmAfterDisableAutosave = function(notReallyUsed) {
        // At this point the user has said go ahead
        gFastecAppMgr.setCameraStateArmed();
    };

    //--------------------------------------
    // Callback from user saying its ok to
    // continue to live mode
    //--------------------------------------
    this.onOKToGoLive = function(isItOK) {
        if (isItOK) {
            // Ask camera to go live
            gFastecAppMgr.setCameraStateLive();

            // Catch state where we are still autosaving!
            if (gCameraData.autoSaveSaving) gFastecAppMgr.changeAppState(fastecAppStates.fastecWaitingForAutoSaving);
        } else {
            // If there is a callback waiting for live status, call it
            // with a false state so it can continue with it's processing thread.
            if (gFastecAppMgr.callMeWhenGoLive != null) gFastecAppMgr.callMeWhenGoLive(false);
        }
    };

    //---------------------------
    // REVIEW Button Click Handler
    //---------------------------
    this.onReviewTBClick = function() {
        // Cannot go to review mode if there isn't data captured
        if (gCameraData.hasCaptureData()) {
            // Ask camera to transition to review.
            gFastecAppMgr.setCameraStateReview();
        }
    };

    //---------------------------
    // TRIGGER Button Click Handler
    //---------------------------
    this.onTriggerTBClick = function() {
        // Allow double trigger - not sure if camera will catch this software double trigger state change request
        if (gCameraData.isCameraInLRAppendMode()) {
            // ask camera to go into triggered mode
            gFastecAppMgr.setCameraStateTriggered();
        } else if (gFastecAppMgr.theState == fastecAppStates.fastecCamArmed) {
            // Ask camera to go into triggered mode
            gFastecAppMgr.setCameraStateTriggered();
        }
    };

    //---------------------------
    // Refresh Camera Status
    //---------------------------
    this.refreshCameraStatus = function() {
        if (gFastecAppMgr.theState == fastecAppStates.fastecUnInit)
            gFastecAppMgr.updateCameraStatus(appmgr_str.app_uninitialized);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecAppPreset)
            gFastecAppMgr.updateCameraStatus(appmgr_str.app_preset);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecAppInit)
            gFastecAppMgr.updateCameraStatus(appmgr_str.app_initializing);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecSysLoadFail)
            gFastecAppMgr.updateCameraStatus(appmgr_str.app_sysLoadFail);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecSysLoaded)
            gFastecAppMgr.updateCameraStatus(appmgr_str.app_sysLoaded);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecCamLive) gFastecAppMgr.updateCameraStatus(appmgr_str.camLive);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecCamArmed)
            gFastecAppMgr.updateCameraStatus(appmgr_str.camArmed);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecCamDeArmed)
            gFastecAppMgr.updateCameraStatus(appmgr_str.camDearmed);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecCamTriggered)
            gFastecAppMgr.updateCameraStatus(appmgr_str.camTrig);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecCamReview)
            gFastecAppMgr.updateCameraStatus(appmgr_str.camReview);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecCamAutosave)
            gFastecAppMgr.updateCameraStatus(appmgr_str.camAutosave);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecCamOther) gFastecAppMgr.updateCameraStatus(appmgr_str.camIdle);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecWaitingToArm)
            gFastecAppMgr.updateCameraStatus(appmgr_str.waitingToArm);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecWaitingToDeArm)
            gFastecAppMgr.updateCameraStatus(appmgr_str.waitingToDeArm);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecWaitingToTrigger)
            gFastecAppMgr.updateCameraStatus(appmgr_str.waitingToTrigger);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecWaitingForLive)
            gFastecAppMgr.updateCameraStatus(appmgr_str.waitingForLive);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecWaitingToReview)
            gFastecAppMgr.updateCameraStatus(appmgr_str.waitingForReviewLoad);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecNetworkGone)
            gFastecAppMgr.updateCameraStatus(appmgr_str.lostNetwork);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecResetToFactory)
            gFastecAppMgr.updateCameraStatus(appmgr_str.app_resetToFactory);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecUpdateCamera)
            gFastecAppMgr.updateCameraStatus(appmgr_str.app_resetToFactory);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecPowerDown)
            gFastecAppMgr.updateCameraStatus(appmgr_str.app_poweringDown);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecBlkRefCalib)
            gFastecAppMgr.updateCameraStatus(appmgr_str.app_blackRefCalib);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecFmtMedia)
            gFastecAppMgr.updateCameraStatus(appmgr_str.app_fmtMediaOp);
        else if (gFastecAppMgr.theState == fastecAppStates.fastecWaitingForAutoSaving)
            gFastecAppMgr.updateCameraStatus(appmgr_str.waitingForAutoSavingComplete);
        else gFastecAppMgr.updateCameraStatus("Status [" + gFastecAppMgr.theState + "]");
    };

    //--------------------------------------
    // Change the camera state to "arm".
    //--------------------------------------
    this.setCameraStateArmed = function() {
        gFastecAppMgr.changeAppState(fastecAppStates.fastecWaitingToArm);
        gFastecAppMgr.changeCameraState(gRegs.GIGE_CAMERA_STATE_ARMED);
    };

    //--------------------------------------
    // Change the camera state to "dearm".
    //--------------------------------------
    this.setCameraStateDeArmed = function() {
        // When we dearm, we also want to stop streaming.
        if (g_fiwss_liveview.is_streaming()) fiLiveViewOffClicked();

        gFastecAppMgr.changeAppState(fastecAppStates.fastecWaitingToDeArm);
        gFastecAppMgr.changeCameraState(gRegs.GIGE_CAMERA_STATE_DEARMING);
    };

    //--------------------------------------
    // Change the camera state to "idle".
    //--------------------------------------
    this.setCameraStateIdle = function() {
        gFastecAppMgr.changeCameraState(gRegs.GIGE_CAMERA_STATE_IDLE);
    };

    //--------------------------------------
    // Change the camera state to "live".
    //--------------------------------------
    this.setCameraStateLive = function() {
        gFastecAppMgr.changeAppState(fastecAppStates.fastecWaitingForLive);
        gFastecAppMgr.changeCameraState(gRegs.GIGE_CAMERA_STATE_LIVE_VIDEO);
    };

    //--------------------------------------
    // Change the camera state to "review".
    //--------------------------------------
    this.setCameraStateReview = function() {
        gFastecAppMgr.changeAppState(fastecAppStates.fastecWaitingToReview);
        gFastecAppMgr.changeCameraState(gRegs.GIGE_CAMERA_STATE_REVIEW);
    };

    //--------------------------------------
    // Change the camera state to "triggered".
    //--------------------------------------
    this.setCameraStateTriggered = function() {
        gFastecAppMgr.changeAppState(fastecAppStates.fastecWaitingToTrigger);
        gFastecAppMgr.changeCameraState(gRegs.GIGE_CAMERA_STATE_TRIGGERED);
    };

    //---------------------------
    // keep the UI synced up for camera state
    //---------------------------
    this.syncToolbarButtons = function() {
        fastecLogToConsole(gLogTrace, "syncToolbarButtons(), sysCameraState = " + gCameraData.sysCameraState);

        var liveViewTBActive = false;

        // Don't update anything if we haven't loaded.
        if (!gFastecAppMgr.systemHasBeenLoaded) return;

        // Did we lose the network?
        if (gFastecAppMgr.theState == fastecAppStates.fastecNetworkGone) {
            fastecLogToConsole(gLogTrace, "syncToolbarButtons: Lost the network?!");
            gToolbarMgr.armTB.setVisibleState(false);
            gToolbarMgr.triggerTB.setVisibleState(false);
            gToolbarMgr.liveTB.setVisibleState(false);
            gToolbarMgr.revTB.setVisibleState(false);
            return;
        }

        // Process based on our state.
        switch (gCameraData.sysCameraState) {
            case gRegs.GIGE_CAMERA_STATE_REVIEW:
                // We're in review, so disable the review button.
                gToolbarMgr.revTB.setActiveState(false);

                // Update the Arm & Trigger buttons. We are not in the armed state, and we don't
                // want to trigger, but it is OK to arm from the review state.
                gToolbarMgr.setArmTrigTBTitles(false, false, true);

                // We can go live.
                gToolbarMgr.liveTB.setActiveState(true);
                break;
            case gRegs.GIGE_CAMERA_STATE_ARMED:
                var showTrigButton = gCameraData.sysCameraState == gRegs.GIGE_CAMERA_STATE_ARMED ? true : false;
                gToolbarMgr.setArmTrigTBTitles(true, showTrigButton, true);

                // We cannot use the Live button to go back. Need to Dearm.
                gToolbarMgr.liveTB.setActiveState(false);

                // Review is not allowed.
                gToolbarMgr.revTB.setActiveState(false);
                break;
            case gRegs.GIGE_CAMERA_STATE_LIVE_VIDEO:
                // We are live, so disable the button.
                gToolbarMgr.liveTB.setActiveState(false);

                // We are not in the armed state, and we don't want to trigger (snapshot), but
                // it is  OK to arm.
                gToolbarMgr.setArmTrigTBTitles(false, false, true);

                // If we have an active partition, we can go into review.
                gToolbarMgr.revTB.setActiveState(gCameraData.actPartState == gRegs.GIGE_PART_STATE_COMPLETE);
                break;
            case gRegs.GIGE_CAMERA_STATE_TRIGGERED:
            case gRegs.GIGE_CAMERA_STATE_IDLE:
            case gRegs.GIGE_CAMERA_STATE_SNAPSHOT:
            case gRegs.GIGE_CAMERA_STATE_AUTOSAVE:
            case gRegs.GIGE_CAMERA_STATE_DEARMING:
                gToolbarMgr.armTB.setActiveState(false);
                gToolbarMgr.liveTB.setActiveState(false);
                gToolbarMgr.revTB.setActiveState(false);
                gToolbarMgr.triggerTB.setActiveState(false);
                break;
        }

        // If we're streaming, show a pause button. Otherwise show a play button.
        gToolbarMgr.setLiveViewToggleTBState(true, !g_fiwss_liveview.is_streaming());
    };

    //---------------------------
    // Camera Status Information
    //---------------------------
    this.updateCameraInfo = function() {
        fastecLogToConsole(gLogTrace, "updateCameraInfo()");

        $("#cameraModelID").html(gCameraData.cameraModel);
        $("#cameraMACID").html(gCameraData.cameraMAC);

        var patch = gCameraData.cameraRev & 0x000000ff;
        var minor = (gCameraData.cameraRev >> 8) & 0x000000ff;
        var major = (gCameraData.cameraRev >> 16) & 0x000000ff;
        var revStr = major + "." + minor + "." + patch;
        $("#cameraRevID").html(revStr);

        patch = gCameraData.cameraFPGARev & 0x0000ffff;
        minor = (gCameraData.cameraFPGARev >> 16) & 0x000000ff;
        major = (gCameraData.cameraFPGARev >> 24) & 0x000000ff;
        revStr = major + "." + minor + "." + patch;
        $("#fpgaVerID").html(revStr);

        $("#sensorVerID").html(gCameraData.cameraSensorRev);
        $("#webappVerID").html(gWebAppVersion);
    };

    //---------------------------
    // Camera Status Update
    //---------------------------
    this.updateCameraStatus = function(theStatusMsg) {
        fastecLogToConsole(gLogTrace, "updateCameraStatus: " + theStatusMsg);
        $("#vrCamState").html(theStatusMsg);
    };

    //---------------------------
    // Camera Record Settings update
    //---------------------------
    this.updateRecSettings = function() {
        var settingsMsg = gCameraData.recROIWidth + "x" + gCameraData.recROIHeight + " @ " + gCameraData.recFPS + "fps";
        $("#vrROIFPS").html(settingsMsg);
        videoSyncWithAR(false);
    };

    //---------------------------
    // Camera State UI Update
    //---------------------------
    this.updateUIForCameraState = function(theCameraStateToUse) {
        fastecLogToConsole(
            gLogTrace,
            "updateUIForCameraState(" + theCameraStateToUse + "), theState = " + gFastecAppMgr.theState
        );

        // Flags to indicate if we want to hide/show a particular element.
        var progbar_hide = true;

        // If state change occurred change webapp state accordingly
        switch (theCameraStateToUse) {
            case gRegs.GIGE_CAMERA_STATE_IDLE:
                gFastecAppMgr.changeAppState(fastecAppStates.fastecCamOther);
                break;
            case gRegs.GIGE_CAMERA_STATE_LIVE_VIDEO:
                // Were we waiting for arming? If so don't pass the live state on down
                // keep waiting
                if (gFastecAppMgr.theState != fastecAppStates.fastecWaitingToArm)
                    gFastecAppMgr.changeAppState(fastecAppStates.fastecCamLive);
                break;
            case gRegs.GIGE_CAMERA_STATE_ARMED:
                progbar_hide = false;
                if (gFastecAppMgr.theState != fastecAppStates.fastecCamArmed)
                    gFastecAppMgr.changeAppState(fastecAppStates.fastecCamArmed);
                break;
            case gRegs.GIGE_CAMERA_STATE_DEARMING:
                progbar_hide = false;
                if (gFastecAppMgr.theState != fastecAppStates.fastecCamDeArmed)
                    gFastecAppMgr.changeAppState(fastecAppStates.fastecCamDeArmed);
                break;
            case gRegs.GIGE_CAMERA_STATE_TRIGGERED:
                progbar_hide = false;
                if (gFastecAppMgr.theState != fastecAppStates.fastecCamTriggered)
                    gFastecAppMgr.changeAppState(fastecAppStates.fastecCamTriggered);
                break;
            case gRegs.GIGE_CAMERA_STATE_REVIEW:
		review_ui_hide = false;
                if (gFastecAppMgr.theState != fastecAppStates.fastecCamReview)
                    gFastecAppMgr.changeAppState(fastecAppStates.fastecCamReview);
                break;
            case gRegs.GIGE_CAMERA_STATE_SNAPSHOT:
                if (gFastecAppMgr.theState != fastecAppStates.fastecCamOther)
                    gFastecAppMgr.changeAppState(fastecAppStates.fastecCamOther);
                break;
            case gRegs.GIGE_CAMERA_STATE_AUTOSAVE:
                gFastecAppMgr.changeAppState(fastecAppStates.fastecCamAutosave);
                break;
	    default:
		break;
        }

        // Are we showing or hiding the progress bar?
        if (progbar_hide) {
            // Hide the progress bar.
            $("#videoCaptureSectionID").hide();
        } else {
            // Make sure the progress bar is visible.
            $("#videoCaptureSectionID").show();

            // Update the current value.
            gArmTrigPctCtrl.setNewCurVal(gCameraData.maxArmFrameNo);
        }

	// Are we showing or hiding the reivew UI?
	if (gVidReviewUI != null) {
	    gVidReviewUI.updateUI();
	}

	// Update the panel.
	if (gPanelMgr != null) {
	    gPanelMgr.updateUI();
	}

	// Update our display.
	resizeKeepingAspectRatio();
    };

    //---------------------------
    // VideoFrame Info Box
    //---------------------------
    this.vfInfoBoxHide = function() {
        $(".vsTextBox").css("z-index", 1);
        $(".vsTextBox").hide();
    };
    this.vfInfoBoxShow = function(theStatusMsg) {
        $(".vsTextBox").css("z-index", 600);
        $("#vfInfoTextID").html(theStatusMsg);
        $(".vsTextBox").show();
    };
}
