//------------------------------------------------
// fastec_wss.js
//
// Copyright 2010-2018 (c) Fastec Imaging as an unpublished work.
// All Rights Reserved.
//
// The information contained herein is confidential property of
// Fastec Imaging. The use, copying, transfer or disclosure of such
// information is prohibited except by express written agreement
// with Fastec Imaging.
//
// Fastec Imaging Camera Control Web Application
// Web Socket Server Interface Functions for Application
//------------------------------------------------

// The global fi_wss_socket object created for the liveview protocol.
// The reason for being global is because using "this" pointer will not
// work when message handlers are invoked as callbacks for the
// socket interface.  There is only one liveview WSS protocol object.
var g_fiwss_liveview = null;

// The global fi_wss_socket object created for the playback protocol.
// The reason for being global is because using "this" pointer will not
// work when message handlers are invoked as callbacks for the
// socket interface.  There is only one playback WSS protocol object.
var g_fiwss_playback = null;

// The default port used by the WSS is 7681.
var g_fiwss_default_port = ":7681";

//------------------------------------------------------------------------------
// All Fastec WSS objects can use this common function to determine
// if the WSS is connected and is ready to use.
//------------------------------------------------------------------------------
function fiwss_is_connected(myWSSConnection) {
    // Look at readyState to see if connection is viable.
    if (myWSSConnection !== null) {
        if (typeof MozWebSocket != "undefined") {
            if (myWSSConnection.readyState == 1) {
                return true;
            }
        } else {
            if (myWSSConnection.readyState == 1) {
                return true;
            }
        }
    }

    return false;
}

//---------------------------
// Build My IP Address Component.
//---------------------------
function fiwss_make_url() {
    var wssURL = "";
    var addrType = "ws://"; // default to http address type

    // Build up a Web Socket URL to our camera.
    wssURL = addrType + gIPPartAddressIs;

    // Not sure what kind of warnings nor which one to use if the gIPPartAddress
    // is unknown.  -- it will default to wireless if dongle set or wired if not.
    return wssURL;
}

//------------------------------------------------------------------------------
//******************************************************************************
//------------------------------------------------------------------------------
// LIVEVIEW SUPPORT
//------------------------------------------------------------------------------
//******************************************************************************
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// fiwss_liveview is the object that manages the connection to Fastec Imaging
// Web Socket Server using the fi-liveview-protocol identifier.  This does
// not start the liveview message strem. It establishes the connection with a
// listener state of "off".
//------------------------------------------------------------------------------
function fiwss_liveview(in_cbhConnected, in_cbhDisconnected) {
    fiwss_live_str = {
        fiwss_lv_connected:
            "Live View Connection Established<br>Please Stand By...",
        fiwss_lv_noconnect:
            "Unable to Make Live View Connection to Server<br>Check Live View Server on Camera<br><br>Try to Refresh Browser to Attempt New Connection.",
        fiwss_lv_disconnected:
            "Lost Connection to Live View Server<br><br>Try to Refresh Browser to Reconnect."
    };

    // Will be not-null if connected.
    this.fiwss_liveview_conn = null;

    // Is the WSS connected and ready to send images.
    this.fiwss_liveview_has_connection = false;

    // Keep a state of when a connection was lost (means it was opened
    // then closed by the remote WSS).
    this.fiwss_liveview_lost_connection = false;

    // Is the WSS streaming liveview images.
    this.fiwss_liveview_is_streaming = false;

    // Using the default port of 7681.
    this.fi_wssIPAddress = fiwss_make_url() + g_fiwss_default_port;

    // This name is used to declare the type of protocol on the WSS connection.
    this.liveview_protocol_type = "fi-liveview-protocol";

    // Save the time of last liveview image displayed.
    this.time_lastimage = 0;

    // What is the size of current image.
    this.image_bytes_toread = 0;

    // Store the callback handler for when the connection is completed.
    this.cbhCallMeWhenConnected = in_cbhConnected;

    // Store the callback handler for when the connection is disconnected.
    this.cbhCallMeWhenDisconnected = in_cbhDisconnected;

    //---------------------------------------------------------------
    // Make a connection to WSS liveview protocol
    //---------------------------------------------------------------
    this.create_wss_socket = function() {
        fastecLogToConsole(gLogTrace, "fiwss_liveview::create_wss_socket()");

        // The type of browser determines the type of socket to create.
        if (typeof MozWebSocket != "undefined") {
            this.fiwss_liveview_conn = new MozWebSocket(
                this.fi_wssIPAddress,
                this.liveview_protocol_type
            );
        } else {
            this.fiwss_liveview_conn = new WebSocket(
                this.fi_wssIPAddress,
                this.liveview_protocol_type
            );
        }

        // We need to declare the event handler callbacks AFTER we've created
        // the socket.
        try {
            //---------------------------------------------------------------
            // Callback function is called when live view connection established
            //---------------------------------------------------------------
            this.fiwss_liveview_conn.onopen = function() {
                fastecLogToConsole(gLogTrace, "fiwss_liveview_conn.onopen()");

                // Note that we're using the global here - "this" won't work.
                g_fiwss_liveview.fiwss_liveview_has_connection = true;
                gFastecAppMgr.vfInfoBoxShow(
                    fiwss_live_str.fiwss_lv_connected,
                    false
                );

                // Make a call to the callback handler for connection status notification.
                g_fiwss_liveview.cbhCallMeWhenConnected();
            };

            //---------------------------------------------------------------
            // Callback function is called when a live view image is received.
            //---------------------------------------------------------------
            this.fiwss_liveview_conn.onmessage = function got_packet(msg) {
                fastecLogToConsole(
                    gLogLiveViewImageLoads,
                    "fiwss_liveview_conn.onmessage()"
                );

                // Should have received a Binary Data -- what to do with it...
                if (msg.data instanceof Blob) {
                    var logMessage = "";

                    // Liveview in the browser's WSS will always send a msg with
                    // the 12 bytes in length representing the next frames size
                    // so if that is received only mark the size expected --
                    if (msg.data.size == 12) {
                        logMessage =
                            "IMAGE HEADER: msg.data.size=" + msg.data.size;
                        fastecLogToConsole(gLogLiveViewImageLoads, logMessage);
                        return; // done with the header so return
                    }

                    // If NOT HEADER Fall through to process imagedata
                    logMessage = "IMAGE DATA: msg.data.size=" + msg.data.size;
                    fastecLogToConsole(gLogLiveViewImageLoads, logMessage);

                    // Render the image and reset the stream info...
                    var img = new Image();

                    // Load our image. When it is done loading, we'll get an "onload" event.
                    img.src = window.URL.createObjectURL(msg.data);

                    // Our "onload" event handler.
                    img.onload = function(e) {
                        // fastecLogToConsole(gLogTrace, "fiwss_liveview_conn.onload()");

                        // THIS IS HOW CANVAS Graphic Images are scaled...
                        //
                        // The only way the canvas draws the image properly is
                        // to have the div element wrapping it sized to match
                        // the proper viewport based on the resize calculations
                        // set during every window resize.  Using the
                        // VideoImage element achieves this in our HTML setup.
                        //
                        // Also make sure that the style width and height for
                        // the Canvas is set to 100% in both width and height.
                        // This is set in the HTML declaration for the canvas.
                        //
                        // Then make sure that the canvas itself has the width
                        // and height of the full image ROI. This will scale and
                        // draw the image properly.

                        // Get the object used for the Video Live View Canvas.
                        var theCanvas = document.getElementById("videoCanvas");

                        // Here is the important part. The style scale is 100%
                        // for both width and height of the canvas object BUT
                        // in order to display full resolution "scaled" within
                        // the viewport associated for the video frame the full
                        // camera ROI settings must be used to define this size
                        // for the canvas graphics manager (TODO test this on all
                        // browsers supporting HTML5
                        theCanvas.width = gCameraData.recROIWidth;
                        theCanvas.height = gCameraData.recROIHeight;

                        // Draw the image onto the drawing object.
                        var ctx = theCanvas.getContext("2d");
                        ctx.drawImage(img, 0, 0);

                        // Now release the image URL - we don't need it anymore.
                        window.URL.revokeObjectURL(img.src);

                        // Release the image object.
                        img = null;

			// Make sure we resize each frame based on the window size.
			resizeKeepingAspectRatio();

                        // And make sure that the video frame is the topmost frame
                        // in this shared space so that the video is what appears for the user.
                        gFastecAppMgr.vfInfoBoxHide();
                        $(".VideoImage").css("z-index", gVideoImgTopZ);
                        g_fiwss_liveview.time_lastimage = getMyCurrentTime(); // milleseconds
                    };

                    // Our error event handler.
                    img.onerror = img.onabort = function() {
                        img = null;
                    };
                }
            };

            //---------------------------------------------------------------
            // this function is called when live view connection disconnected
            //---------------------------------------------------------------
            this.fiwss_liveview_conn.onclose = function() {
                fastecLogToConsole(gLogTrace, "fiwss_liveview_conn.onclose()");

                // Did we ever have a connection?
                if (g_fiwss_liveview.fiwss_liveview_has_connection)
                    g_fiwss_liveview.fiwss_liveview_lost_connection = true; // we lost a connected WSS
                else g_fiwss_liveview.fiwss_liveview_lost_connection = false; // we never had an open connection

                g_fiwss_liveview.fiwss_liveview_has_connection = false;
                g_fiwss_liveview.fiwss_liveview_is_streaming = false; // not streaming anymore.
                g_fiwss_liveview.fiwss_liveview_conn = null;

                if (g_fiwss_liveview.fiwss_liveview_lost_connection)
                    gFastecAppMgr.vfInfoBoxShow(
                        fiwss_live_str.fiwss_lv_disconnected,
                        false
                    );
                else
                    gFastecAppMgr.vfInfoBoxShow(
                        fiwss_live_str.fiwss_lv_noconnect,
                        false
                    );

                // Hide our image.
                $("#VideoDivImg").hide();

                // Invoke callback handler to notify wss is disconnected
                g_fiwss_liveview.cbhCallMeWhenDisconnected();
            };
        } catch (exception) {
            alert("WSS Liveview Error: " + exception);
        }
    };

    //---------------------------------------------------------------
    // Closes the websocket connection
    //---------------------------------------------------------------
    this.closeWSS = function() {
        fastecLogToConsole(gLogTrace, "fiwss_liveview::closeWSS()");

        if (this.fiwss_liveview_conn !== null) this.fiwss_liveview_conn.close();
    };

    //---------------------------------------------------------------
    // Attempt to reconnect with the WSS liveview protocol
    //---------------------------------------------------------------
    this.reconnect = function() {
        fastecLogToConsole(gLogTrace, "fiwss_liveview::reconnect()");

        if (this.fiwss_liveview_lost_connection) {
            this.make_connection(this);

            // If not able to make connection object report the problem.
            if (this.fiwss_liveview_conn === null)
                gFastecAppMgr.vfInfoBoxShow(
                    fiwss_live_str.fiwss_lv_noconnect,
                    false
                );
        }
    };

    //---------------------------------------------------------------
    // Is the liveview ready for communications?
    //---------------------------------------------------------------
    this.is_connected = function() {
        fastecLogToConsole(gLogTrace, "fiwss_liveview::is_connected()");

        if (fiwss_is_connected(this.fiwss_liveview_conn)) {
            // Make sure the on message handler returned...
            if (this.fiwss_liveview_has_connection) return true;
        }
    };

    //---------------------------------------------------------------
    // Is the liveview wss connection streaming images?
    //---------------------------------------------------------------
    this.is_streaming = function() {
        // Do we have a connection?
        if (this.fiwss_liveview_conn !== null) {
            // Are we streaming?
            if (this.fiwss_liveview_is_streaming) return true;
        }

        return false;
    };

    //---------------------------------------------------------------
    // Was the liveview connection lost by remote WSS?
    //---------------------------------------------------------------
    this.lost_connection = function() {
        if (this.fiwss_liveview_conn !== null) {
            if (this.fiwss_liveview_lost_connection) return true;
        }

        return false;
    };
}

//---------------------------------------------------
// Handler for a turning on a live view stream
//---------------------------------------------------
function fiLiveViewOnClicked() {
    fastecLogToConsole(gLogTrace, "fiLiveViewOnClicked");

    if (g_fiwss_liveview.is_connected()) {
        $("#VideoDivImg").show(); // make sure live viewport is visible
        var jsonLiveViewOn = '{"liveview":"ON"}';
        g_fiwss_liveview.fiwss_liveview_conn.send(jsonLiveViewOn);
        g_fiwss_liveview.fiwss_liveview_is_streaming = true;
        return true;
    }

    return false; // let caller know we didn't send this
}

//---------------------------------------------------
// Handler for a turning off a live view stream
//---------------------------------------------------
function fiLiveViewOffClicked() {
    fastecLogToConsole(gLogTrace, "fiLiveViewOffClicked");

    // if there is no connection to the fi_wss attempt to make one
    if (g_fiwss_liveview.is_connected()) {
        var jsonLiveViewOff = '{"liveview":"OFF"}';
        g_fiwss_liveview.fiwss_liveview_conn.send(jsonLiveViewOff);
        g_fiwss_liveview.fiwss_liveview_is_streaming = false;
        return true;
    }

    return false; // let caller know we didn't send this.
}

//------------------------------------------------------------------------------
//******************************************************************************
//------------------------------------------------------------------------------
// PLAYBACK SUPPORT
//------------------------------------------------------------------------------
//******************************************************************************
//------------------------------------------------------------------------------
// constant state values for missed play operation requests
var kMissedNoPlayOps = 0; // Didn't miss any play requests
var kMissedPlayOpSS = 1; // Missed a SS play request
var kMissedPlayOpStream = 2; // Missed a Stream play request

//------------------------------------------------------------------------------
// fiwss_playback is the object that manages the connection to Fastec Imaging
// Web Socket Server using the fi-playback-protocol identifier.  This does
// not start playback messages, rather it only establishes the connection with
// the web socket that supplies playback functionality.
//
// Expects a callback handler for connection and disconnection states
//------------------------------------------------------------------------------
function fiwss_playback() {
    fiwss_pb_str = {
        fiwss_pb_errTitle: "Image Playback Error",
        fiwss_pb_connected:
            "Image Playback Connection Established<br>Please Stand By...",
        fiwss_pb_noconnect:
            "Unable to Make Playback Connection to Server<br><br>Check Playback Server on Camera<br><br>Try to Refresh Browser to Attempt New Connection.",
        fiwss_pb_imgread_delay:
            "Too much time has elapsed since playback frame was received.<br><br>Check Playback Server on Camera<br><br>Try to Refresh Browser to Attempt New Connection.",
        fiwss_pb_disconnected:
            "Lost Connection to Playback Server<br><br>Try to Refresh Browser to Reconnect.",
        fiwss_pb_busy:
            "Unable to Service Playback Command<br><br>Playback Server on Camera Is Busy<br><br>Another Player May Have Exclusive Control."
    };

    // Will be not-null if connected.
    this.fiwss_playback_conn = null;

    // Is the WSS connected and ready to send images.
    this.fiwss_playback_has_connection = false;

    // Keep a state of when a connection was lost (means it was opened
    // then closed by the remote WSS).
    this.fiwss_playback_lost_connection = false;

    this.waitForPlayStart = false;
    this.waitForPlayStop = false;
    this.playOneFrame = false; // will be true when play start attempted for one frame
    this.playFrames = false; // will be true when play start is attempted for stream
    this.isPlaying = false; // set to true when the play engine thinks it should be playing frames

    // If user wants a play operation but cannot service it this will
    // keep track of the last request so that if necessary it can be
    // serviced when available and stores call state
    this.missedPlayOp = kMissedNoPlayOps; // when a play request is made but cannot be serviced this stores it
    this.missedOneFrame = false; // state of one Frame for missed Play Op
    this.missedGoingForward = false; // true if playing forward
    this.missedFrameId = -1; // zero-based frame or -1 if use current
    this.missedCutIn = -1; // zero-based frame start of cut or -1 if none
    this.missedCutOut = -1; // zero-based frame end of cut or -1 if none
    this.missedFrameRate = 0; // how fast to playback frames
    this.missedCBHDisconnected = 0; // callback if disconnect occurs
    this.missedCBHDisplay = 0;

    // Using the default port of 7681.
    this.fi_wssIPAddress = fiwss_make_url() + g_fiwss_default_port;

    // This name is used to declare the type of protocol on the WSS connection.
    this.playback_protocol_type = "fi-playback-protocol";

    // Store the callback handler for when connection is completed
    this.cbhCallMeWhenConnected = null;

    // Store the callback handler for when connection is disconnected.
    this.cbhCallMeWhenDisconnected = null;

    // Store the callback handler for display managment when playback
    // frame is received by the playback protocol.
    this.cbhCallMeWhenPBFrameArrives = null;

    // When an image frame is available for playback, these values will
    // hold the meta information of the frame.
    this.imgSize = -1;
    this.imgFrame = -1;
    this.imgLastTimeRead = 0; // when did we last read a playback frame

    //--------------------------------------
    // Callback Handler - this is called once
    // the user has acknowledged that the playback
    // engine is disabled and most likely stalled
    //--------------------------------------
    this.cbhPBStalledAcknowledged = function(dontCare) {
        // The user has left the final dialog but the camera may not
        // be ready to reset.  Need to set a state of waiting.
        gFastecAppMgr.onNetworkLostConn();

        // Force a close and stop state.
        g_fiwss_playback.closeWSS();
    };

    //---------------------------------------------------------------
    // Provide way to test for extraordinary elapsed time when the
    // play engine thinks it's playing and no images are being read
    //---------------------------------------------------------------
    this.check_wss_playstatus = function() {
        // if we think we are playing image data check for a delay that
        // seems too long so that the play engine can be marked stopped
        // and a message displayed to the user
        if (g_fiwss_playback.playFrames) {
            var timeNow = getMyCurrentTime(); // milleseconds

            var elapsedTime = timeNow - g_fiwss_playback.imgLastTimeRead;
            if (elapsedTime >= 7000) {
                // If a liveview image has not been received in 7 seconds report problem
                var mb = new fastecMessageBox();
                g_fiwss_playback.cbhSetPBStop(true); // reset as if stop button pressed
                mb.showMessageBoxWithClose(
                    fiwss_pb_str.fiwss_pb_errTitle,
                    fiwss_pb_str.fiwss_pb_imgread_delay,
                    g_fiwss_playback.cbhPBStalledAcknowledged,
                    false
                );
            }
        }
    };

    //---------------------------------------------------------------
    // Make a connection object to use for WSS playback protocol
    //---------------------------------------------------------------
    this.create_wss_socket = function() {
        fastecLogToConsole(gLogPBStateInfo, "fiwss_playback::create_wss_socket()");

        // Sanity check: if already connected cannot connect again.
        if (this.fiwss_playback_conn !== null) {
	    return; // already has connect object don't create it again
	}

        // Create the socket connection depending on the type of browser.
        if (typeof MozWebSocket != "undefined") {
            this.fiwss_playback_conn = new MozWebSocket(
                this.fi_wssIPAddress,
                this.playback_protocol_type
            );
        } else {
            this.fiwss_playback_conn = new WebSocket(
                this.fi_wssIPAddress,
                this.playback_protocol_type
            );
        }

        // We need to declare the event handler callbacks AFTER we've created
        // the socket.
        try {
            //---------------------------------------------------------------
            // Callback function is called when playback connection established.
            //---------------------------------------------------------------
            this.fiwss_playback_conn.onopen = function() {
                // log the connection
                fastecLogToConsole(gLogPBStateInfo, "Playback Connection Opened");

                // Note that we're using the global here - "this" won't work.
                g_fiwss_playback.fiwss_playback_has_connection = true;
                gFastecAppMgr.vfInfoBoxShow(
                    fiwss_pb_str.fiwss_pb_connected,
                    false
                );

                // Make a call to the cbh for connection status notification
                if (g_fiwss_playback.cbhCallMeWhenConnected !== null)
                    g_fiwss_playback.cbhCallMeWhenConnected();
            };

            //---------------------------------------------------------------
            // Callback function is called when playback image is ready for display.
            //---------------------------------------------------------------
            this.fiwss_playback_conn.onmessage = function got_packet(msg) {
                fastecLogToConsole(gLogPBStateInfo, "Playback Image Draw: msg.data.size(" + msg.data.size + ")");

                // Should have received a Binary Data -- what to do with it...
                if (msg.data instanceof Blob) {
                    // Playback in the browser's WSS will always send a msg with
                    // the 12 bytes in length representing the next frames size
                    // so if that is received only mark the size expected --
                    if (msg.data.size == 12) {
                        logMessage = "IMAGE HEADER: msg.data.size=" + msg.data.size;
                        fastecLogToConsole(gLogPBStateInfo, logMessage);

			var myReader = new FileReader();
			myReader.onloadend  = function() {
			    var dataMagic = new DataView(myReader.result,0,4);
			    var dataSize = new DataView(myReader.result,4,4);
			    var dataID = new DataView(myReader.result,8,4);
			    g_fiwss_playback.imgSize = dataSize.getInt32(0).toString(16);
			    g_fiwss_playback.imgFrame = dataID.getInt32(0).toString();
			};
			myReader.readAsArrayBuffer(msg.data);
     
                        logMessage = "IMAGE HEADER: imgSize=" + g_fiwss_playback.imgSize + ", imgFrame=" + g_fiwss_playback.imgFrame;
                        fastecLogToConsole(gLogPBStateInfo, logMessage);

                        return; // done with the header so return
                    }

                    // If NOT HEADER Fall through to process imagedata
                    logMessage = "IMAGE DATA: msg.data.size=" + msg.data.size;
                    fastecLogToConsole(gLogPBStateInfo, logMessage);
		
		    // Create a new image object and have the browser load it.
		    var img = new Image();
		    img.src = window.URL.createObjectURL(msg.data);

		    // ----------------------------------------
		    // Handle the onload event when the browser loads the image.
		    // ----------------------------------------
		    img.onload = function(e) {
			fastecLogToConsole(gLogPBStateInfo, "fiwss_playback_conn.onload()");

			// If there is a pending stop request ignore the image.
			if (g_fiwss_playback.waitForPlayStop) {
			    return;
			}

			// THIS IS HOW CANVAS Graphic Images are scaled...
			//
			// the only way the canvas draws the image properly is
			// to have the div element wrapping it sized to match
			// the proper viewport based on the resize calculations
			// set during every window resize.  Using the
			// VideoImg element achieves this in our html setup.
			//
			// Also make sure that the style width and height for
			// the Canvas is set to 100% in both width and height
			// this was set in the html declaration for the canvas
			//
			// then make sure that the canvas itself has the width
			// and height of the full image roi this will scale and
			// draw the image properly set here below

			// get the object used for the Video Live View Canvas
			var theCanvas = document.getElementById("videoCanvas");

			// Here is the important part. The style scale is 100%
			// for both width and height of the canvas object BUT
			// in order to display full resolution "scaled" within
			// the viewport associated for the video frame the full
			// camera ROI settings must be used to define this size
			// for the canvas graphics manager (TODO test this on all
			// browsers supporting HTML5).
			theCanvas.width = gCameraData.recROIWidth;
			theCanvas.height = gCameraData.recROIHeight;

			var ctx = theCanvas.getContext("2d");
			ctx.drawImage(img, 0, 0);

			// Now release the image URL - we don't need it anymore.
			window.URL.revokeObjectURL(img.src);

			// Release the image object.
			img = null;

			// Make sure we resize each frame based on the window size.
			resizeKeepingAspectRatio();

			// The callback will handle updating the metadata for the new frame.
			if (g_fiwss_playback.cbhCallMeWhenPBFrameArrives !== null)
			    g_fiwss_playback.cbhCallMeWhenPBFrameArrives(g_fiwss_playback.imgFrame);

			fastecLogToConsole(
			    gLogPBStateInfo,
			    "Playback Image Draw: FrameId(" +
				g_fiwss_playback.imgFrame +
				")" +
				"OneFrame(" +
				g_fiwss_playback.playOneFrame +
				")"
			);

			// And make sure that the video frame is the topmost frame
			// in this shared space so that the video is what appears for the user.
			gFastecAppMgr.vfInfoBoxHide();
                        $(".VideoImage").css("z-index", gVideoImgTopZ);
			g_fiwss_playback.imgLastTimeRead = getMyCurrentTime(); // milleseconds
		    };

		    // ----------------------------------------
		    // Handle any errors by discarding the image.
		    // ----------------------------------------
		    img.onerror = img.onabort = function() {
			img = null;
		    };

		    // Reset the one frame as having been played.
		    if (g_fiwss_playback.playOneFrame)
			g_fiwss_playback.playOneFrame = false;
                }
            };

            //---------------------------------------------------------------
            // this function is called when playback connection disconnected
            //---------------------------------------------------------------
            this.fiwss_playback_conn.onclose = function() {
                fastecLogToConsole(
                    gLogPBStateInfo,
                    "Playback onclose: fiwss_playback_has_connection(" +
                        this.fiwss_playback_has_connection +
                        ")"
                );

                // did we ever have a connection?
                if (g_fiwss_playback.fiwss_playback_has_connection)
                    g_fiwss_playback.fiwss_playback_lost_connection = true; // we lost a connected WSS
                else g_fiwss_playback.fiwss_playback_lost_connection = false; // we never had an open connection

                g_fiwss_playback.fiwss_playback_has_connection = false;
                g_fiwss_playback.fiwss_playback_conn = null;

                // TODO - what if waiting on FCP command when wss disconnects?
                // it shouldn't matter.  but can get in a state where stuck?

                if (g_fiwss_playback.fiwss_playback_lost_connection)
                    gFastecAppMgr.vfInfoBoxShow(
                        fiwss_pb_str.fiwss_pb_disconnected,
                        false
                    );
                else
                    gFastecAppMgr.vfInfoBoxShow(
                        fiwss_pb_str.fiwss_pb_noconnect,
                        false
                    );

                // hide our image.
                $("#VideoDivImg").hide();

                // invoke callback handler to notify wss is disconnected
                if (g_fiwss_playback.cbhCallMeWhenDisconnected !== null)
                    g_fiwss_playback.cbhCallMeWhenDisconnected();
            };
        } catch (exception) {
            alert("WSS Playback Error: " + exception);
        }
    };

    //---------------------------------------------------------------
    // Closes the websocket connection
    //---------------------------------------------------------------
    this.closeWSS = function() {
        fastecLogToConsole(
            gLogPBStateInfo,
            "Playback closeWSS: fiwss_playback_conn(" +
                g_fiwss_playback.fiwss_playback_conn +
                ")"
        );
        if (this.fiwss_playback_conn !== null) this.fiwss_playback_conn.close();
    };

    //---------------------------------------------------------------
    // Attempt to reconnect with the WSS playback protocol
    //---------------------------------------------------------------
    this.reconnect = function() {
        fastecLogToConsole(
            gLogPBStateInfo,
            "Playback reconnect: fiwss_playback_lost_connection(" +
                g_fiwss_playback.fiwss_playback_lost_connection +
                ")"
        );

        if (this.fiwss_playback_lost_connection) {
            // the wss socket was closed and must be recreated
            this.create_wss_socket();

            // if not able to make connection object report the problem.
            if (this.fiwss_playback_conn === null)
                gFastecAppMgr.vfInfoBoxShow(
                    fiwss_pb_str.fiwss_pb_noconnect,
                    false
                );
        }
    };

    //---------------------------------------------------------------
    // This function is called with result of the setPlaybackMode
    // call to enable or disable playback operations.
    //---------------------------------------------------------------
    this.cbhSetPBStart = function(status) {
        fastecLogToConsole(
            gLogPBStateInfo,
            "Playback cbhSetPBStart: waitForPlayStart(" +
                g_fiwss_playback.waitForPlayStart +
                ")" +
                "missedPlayOp(" +
                g_fiwss_playback.missedPlayOp +
                ")" +
                "waitForPlayStop(" +
                g_fiwss_playback.waitForPlayStop +
                ")" +
                "status(" +
                status +
                ")" +
                "isPlaying(" +
                g_fiwss_playback.isPlaying +
                ")" +
                "playOneFrame(" +
                g_fiwss_playback.playOneFrame +
                ")" +
                "playFrames(" +
                g_fiwss_playback.playFrames +
                ")"
        );

        g_fiwss_playback.waitForPlayStart = false;

        // if it failed we can't be playing...
        if (!status) {
            gFastecAppMgr.vfInfoBoxShow(fiwss_pb_str.fiwss_pb_busy, false);
            // reset the internal play status information
            g_fiwss_playback.playOneFrame = false;
            g_fiwss_playback.playFrames = false;
        } else {
            g_fiwss_playback.isPlaying = true;

            // if missed a playback operation see if can start it...
            if (
                g_fiwss_playback.missedPlayOp == kMissedPlayOpSS ||
                g_fiwss_playback.missedPlayOp == kMissedPlayOpStream
            ) {
                // clear the missed state and try again
                g_fiwss_playback.missedPlayOp = kMissedNoPlayOps;
                g_fiwss_playback.start_playback(
                    g_fiwss_playback.missedOneFrame,
                    g_fiwss_playback.missedGoingForward,
                    g_fiwss_playback.missedFrameId,
                    g_fiwss_playback.missedCutIn,
                    g_fiwss_playback.missedCutOut,
                    g_fiwss_playback.missedFrameRate,
                    g_fiwss_playback.missedCBHDisconnected,
                    g_fiwss_playback.missedCBHDisplay
                );
            }
        }
    };

    //---------------------------------------------------------------
    // Provide an interface to start playback
    //---------------------------------------------------------------
    this.start_playback = function(
        in_oneFrame,
        in_GoingForward, // true if playing forward
        in_FrameId, // zero-based frame or -1 if use current
        in_CutIn, // zero-based frame start of cut or -1 if none
        in_CutOut, // zero-based frame end of cut or -1 if none
        in_FrameRate, // how fast to playback frames
        in_cbhDisconnected, // callback if disconnect occurs
        in_cbhDisplay // callback for handling image display
    ) {
        fastecLogToConsole(
            gLogPBStateInfo,
            "Playback start_playback: waitForPlayStart(" +
                g_fiwss_playback.waitForPlayStart +
                ")" +
                "missedPlayOp(" +
                g_fiwss_playback.missedPlayOp +
                ")" +
                "waitForPlayStop(" +
                g_fiwss_playback.waitForPlayStop +
                ")" +
                "isPlaying(" +
                g_fiwss_playback.isPlaying +
                ")" +
                "in_FrameId(" +
                in_FrameId +
                ")" +
                "in_oneFrame(" +
                in_oneFrame +
                ")"
        );

        var sentRequest = false;
        g_fiwss_playback.missedPlayOp = kMissedNoPlayOps; // always reset this state

        g_fiwss_playback.imgLastTimeRead = getMyCurrentTime(); // milleseconds

        // can only attempt to play if have a connection
        if (this.is_connected()) {
            // make sure the camera is in the right state to start
            // playback functionality.
            if (gCameraData.isReviewPartitionActive()) {
                if (!g_fiwss_playback.waitForPlayStart) {
                    g_fiwss_playback.waitForPlayStart = true;

                    // single step or stream?
                    if (in_oneFrame) {
                        g_fiwss_playback.playOneFrame = true;
                        // write to the camera to enable the playback
                        gCameraData.rvwSSPlayback(
                            in_FrameId,
                            this.cbhSetPBStart
                        );
                    } else {
                        g_fiwss_playback.playFrames = true;
                        // write to the camera to enable the playback
                        gCameraData.rvwStartPlayback(
                            in_GoingForward,
                            in_FrameId,
                            in_CutIn,
                            in_CutOut,
                            in_FrameRate,
                            this.cbhSetPBStart
                        );
                        // start an initial time stamp
                        g_fiwss_playback.imgLastTimeRead = getMyCurrentTime(); // milleseconds
                    }
                    sentRequest = true;
                } else {
                    // missed this request so save its state
                    if (in_oneFrame)
                        g_fiwss_playback.missedPlayOp = kMissedPlayOpSS;
                    else g_fiwss_playback.missedPlayOp = kMissedPlayOpStream;

                    // save the state so can attempt to reset it last playback op completes
                    g_fiwss_playback.missedOneFrame = in_oneFrame;
                    g_fiwss_playback.missedGoingForward = in_GoingForward;
                    g_fiwss_playback.missedFrameId = in_FrameId;
                    g_fiwss_playback.missedCutIn = in_CutIn;
                    g_fiwss_playback.missedCutOut = in_CutOut;
                    g_fiwss_playback.missedFrameRate = in_FrameRate;
                    g_fiwss_playback.missedCBHDisconnected = in_cbhDisconnected;
                    g_fiwss_playback.missedCBHDisplay = in_cbhDisplay;

                    fastecLogToConsole(
                        gLogPBStateInfo,
                        "Playback start_playback MISSED STATE TRUE: waitForPlayStart(" +
                            g_fiwss_playback.waitForPlayStart +
                            ")" +
                            "missedPlayOp(" +
                            g_fiwss_playback.missedPlayOp +
                            ")" +
                            "waitForPlayStop(" +
                            g_fiwss_playback.waitForPlayStop +
                            ")" +
                            "isPlaying(" +
                            g_fiwss_playback.isPlaying +
                            ")" +
                            "playOneFrame(" +
                            g_fiwss_playback.playOneFrame +
                            ")" +
                            "playFrames(" +
                            g_fiwss_playback.playFrames +
                            ")"
                    );
                }
            }
        }
        return sentRequest; // sometimes overlap can occur - let user know if request sent
    };

    //---------------------------------------------------------------
    // This function is called with result of the setPlaybackMode
    // call to enable or disable playback operations.
    //---------------------------------------------------------------
    this.cbhSetPBStop = function(status) {
        fastecLogToConsole(
            gLogPBStateInfo,
            "Playback cbhSetPBStop: waitForPlayStart(" +
                g_fiwss_playback.waitForPlayStart +
                ")" +
                "status(" +
                status +
                ")" +
                "missedPlayOp(" +
                g_fiwss_playback.missedPlayOp +
                ")" +
                "isPlaying(" +
                g_fiwss_playback.isPlaying +
                ")" +
                "playOneFrame(" +
                g_fiwss_playback.playOneFrame +
                ")" +
                "playFrames(" +
                g_fiwss_playback.playFrames +
                ")"
        );

        g_fiwss_playback.waitForPlayStop = false;
        g_fiwss_playback.playOneFrame = false;
        g_fiwss_playback.playFrames = false;
        g_fiwss_playback.isPlaying = false;

        // if it failed we can't be playing...
        if (!status) {
            gFastecAppMgr.vfInfoBoxShow(fiwss_pb_str.fiwss_pb_busy, false);
        }
    };

    //---------------------------------------------------------------
    // Provide an interface to stop any playback
    //---------------------------------------------------------------
    this.stop_playback = function() {
        fastecLogToConsole(
            gLogPBStateInfo,
            "Playback stop_playback: waitForPlayStart(" +
                g_fiwss_playback.waitForPlayStart +
                ")" +
                "missedPlayOp(" +
                g_fiwss_playback.missedPlayOp +
                ")" +
                "waitForPlayStop(" +
                g_fiwss_playback.waitForPlayStop +
                ")" +
                "isPlaying(" +
                g_fiwss_playback.isPlaying +
                ")" +
                "playOneFrame(" +
                g_fiwss_playback.playOneFrame +
                ")" +
                "playFrames(" +
                g_fiwss_playback.playFrames +
                ")"
        );

        // the user has stopped playback remove any missed play state
        g_fiwss_playback.missedPlayOp = kMissedNoPlayOps;

        // if do not have a connection then establish one.
        if (this.is_connected()) {
            // if camera has playback actually enabled mark it to be off
            if (gCameraData.pbEnabled) {
                if (!g_fiwss_playback.waitForPlayStop) {
                    g_fiwss_playback.waitForPlayStop = true;

                    // write to the camera to enable the playback
                    gCameraData.rvwStopPlayback(g_fiwss_playback.cbhSetPBStop);
                }
            }
        }
    };

    //---------------------------------------------------------------
    // Is the playback ready for communications?
    //---------------------------------------------------------------
    this.is_connected = function() {
        return fiwss_is_connected(this.fiwss_playback_conn);
    };

    //---------------------------------------------------------------
    // Was the connected playback connection lost by remote WSS?
    //---------------------------------------------------------------
    this.lost_connection = function() {
        fastecLogToConsole(
            gLogPBStateInfo,
            "Playback Lost Connection: fiwss_playback_conn(" +
                this.fiwss_playback_conn +
                ")"
        );

        // make the connection to WSS for liveview communications
        if (this.fiwss_playback_conn !== null) {
            if (this.fiwss_playback_lost_connection) return true;
        }
        return false;
    };
}

//------------------------------------------------------------------------------
//******************************************************************************
//------------------------------------------------------------------------------
// FCP SUPPORT
//------------------------------------------------------------------------------
//******************************************************************************
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// fiwss_fcp is the object that manages the connection to Fastec Imaging
// Web Socket Server using either the fi-fcp-protocol or fi-fcp-listener-protocol
// identifier.  These FCP protocols either read/write or listen for changes
// to the GIGE FCP registers on the camera
//
// expects a callback handler for connection and disconnection states
//------------------------------------------------------------------------------
var gInstanceNbr = 1;
function fiwss_fcp(
    in_useAsListener, // using this connection for listener purposes?
    in_cbhConnected, // cbh when connection is confirmed
    in_cbhDisconnected // cbh when disconnection is confirmed
) {
    fiwss_fcp_str = {
        fiwss_fcp_connected: "FCP Connection Established<br>Please Stand By...",
        fiwss_fcp_noconnect:
            "Unable to Make FCP Connection to Server<br><br>Check FCP Server on Camera<br><br>Try to Refresh Browser to Attempt New Connection.",
        fiwss_fcp_disconnected:
            "Lost Connection to FCP Server<br><br>Try to Refresh Browser to Reconnect."
    };

    // need to find the object instance when callback events are invoked
    this.instance = gInstanceNbr;
    gInstanceNbr++;

    // We need a reference back to "this" object. We use this in the body
    // of the event handlers defined below.
    var static_handle = this;

    // Our protocol types.
    var kFCPLoadStore = "fi-fcp-protocol";
    var kFCPListen = "fi-fcp-listener-protocol";

    // the protocol type will change depending on the
    // in_useAsListener state where 0 means no and 1 means yes
    this.fcp_protocol_type = kFCPLoadStore;

    // How is the connection used: as a listener or as a FCP load and store?
    if (in_useAsListener == 1) this.fcp_protocol_type = kFCPListen;

    // code responses from send request
    this.sendNoConnection = -1;
    this.sendWaitingOnPastRequest = 2;
    this.sendSentRequest = 1;

    // the json result value will be stored in an instance variable
    this.jsonResult = null;

    // the connection socket - null when not connected
    this.fiwss_fcp_conn = null;

    // is the WSS connected and ready to send and receive fcp requests
    this.fiwss_fcp_has_connection = false;

    // keep a state of when a connection was lost (means it was opened
    // then closed by the remote WSS)
    this.fiwss_fcp_lost_connection = false;

    // using the default port of 7681
    this.fi_wssIPAddress = fiwss_make_url() + g_fiwss_default_port;

    // store the callback handler for when connection is completed
    this.cbhCallMeWhenConnected = in_cbhConnected;

    // store the callback handler for when connection is disconnected
    this.cbhCallMeWhenDisconnected = in_cbhDisconnected;

    // store the callback handler for when the json response package is
    // received on the wss socket
    this.cbhCallMeWhenResponsed = null;

    // keep a state of when a request is pending
    this.fiwss_fcp_requesting = false;

    // keep a copy of the data package being sent on socket
    this.requestDataPkg = null;

    //---------------------------------------------------------------
    // Make a connection object to use for WSS FCP protocol
    //---------------------------------------------------------------
    this.create_wss_socket = function() {
        // If already connected cannot connect again.
        if (this.fiwss_fcp_conn !== null) return; // already has connect object don't create it again

        // Depending on the type of browser determines the type of socket to create.
        if (typeof MozWebSocket != "undefined") {
            this.fiwss_fcp_conn = new MozWebSocket(
                this.fi_wssIPAddress,
                this.fcp_protocol_type
            );
        } else {
            this.fiwss_fcp_conn = new WebSocket(
                this.fi_wssIPAddress,
                this.fcp_protocol_type
            );
        }

        // These are the WSS handlers for web socket FCP interfaces used in browser.
        // They must be set at this point after we've actually created the socket.
        //
        // NOTE: We can't use "this" in the event handlers, because that won't be
        // 	 the correct "this". So we have our "static_handle" to get back to
        // 	 our real instance.
        try {
            //---------------------------------------------------------------
            // callback function is called when FCP connection established
            //---------------------------------------------------------------
            this.fiwss_fcp_conn.onopen = function() {
                static_handle.fiwss_fcp_has_connection = true;
                gFastecAppMgr.vfInfoBoxShow(
                    fiwss_fcp_str.fiwss_fcp_connected,
                    false
                );

                // make a call to the cbh for connection status notification
                if (static_handle.cbhCallMeWhenConnected !== null)
                    static_handle.cbhCallMeWhenConnected();
            };

            //---------------------------------------------------------------
            // callback function is called when FCP message is received from WSS
            //---------------------------------------------------------------
            this.fiwss_fcp_conn.onmessage = function got_packet(msg) {
                if (static_handle.requestDataPkg !== null) {
                    // The parse should pull out the object data into the
                    // request data package in usable form
                    static_handle.requestDataPkg.jsonResponseData = JSON.parse(
                        msg.data,
                        function(key, value) {
                            var type;
                            if (value && typeof value === "object") {
                                type = value.type;
                                if (
                                    typeof type === "string" &&
                                    typeof window[type] === "function"
                                ) {
                                    return new window[type](value);
                                }
                            }
                            return value;
                        }
                    );

                    // Make sure to note that this request has been serviced.
                    static_handle.fiwss_fcp_requesting = false;

                    // Call the callback handler.
                    if (static_handle.cbhCallMeWhenResponsed !== null)
                        static_handle.cbhCallMeWhenResponsed(
                            static_handle.requestDataPkg
                        );
                }
            };

            //---------------------------------------------------------------
            // This function is called when playback connection disconnected
            //---------------------------------------------------------------
            this.fiwss_fcp_conn.onclose = function() {
                // Did we ever have a connection?
                if (static_handle.fiwss_fcp_has_connection)
                    static_handle.fiwss_fcp_lost_connection = true; // we lost a connected WSS
                else static_handle.fiwss_fcp_lost_connection = false; // we never had an open connection

                static_handle.fiwss_fcp_has_connection = false;
                static_handle.fiwss_fcp_conn = null;

                // TODO - what if waiting on FCP command when WSS disconnects?
                // it shouldn't matter.  but can get in a state where stuck?

                if (static_handle.fiwss_fcp_lost_connection)
                    gFastecAppMgr.vfInfoBoxShow(
                        fiwss_fcp_str.fiwss_fcp_disconnected,
                        false
                    );
                else
                    gFastecAppMgr.vfInfoBoxShow(
                        fiwss_fcp_str.fiwss_fcp_noconnect,
                        false
                    );

                // Invoke callback handler to notify wss is disconnected.
                if (static_handle.cbhCallMeWhenDisconnected !== null)
                    static_handle.cbhCallMeWhenDisconnected();
            };
        } catch (exception) {
            alert("WSS FCP Error: " + exception);
        }
    };

    //---------------------------------------------------------------
    // Closes the websocket connection
    //---------------------------------------------------------------
    this.closeWSS = function() {
        if (this.fiwss_fcp_conn !== null) this.fiwss_fcp_conn.close();
    };

    //---------------------------------------------------------------
    // Attempt to reconnect with the WSS fcp protocol
    //---------------------------------------------------------------
    this.reconnect = function() {
        if (this.fiwss_fcp_lost_connection) {
            // The wss socket was closed and must be recreated.
            this.create_wss_socket();

            // If not able to make connection object report the problem.
            if (this.fiwss_fcp_conn === null)
                gFastecAppMgr.vfInfoBoxShow(
                    fiwss_fcp_str.fiwss_fcp_noconnect,
                    false
                );
        }
    };

    //---------------------------------------------------------------
    // Is the fcp being used as a listener?
    //---------------------------------------------------------------
    this.is_listener = function() {
        return this.fcp_protocol_type == this.kFCPListen;
    };

    //---------------------------------------------------------------
    // Is the fcp ready for communications?
    //---------------------------------------------------------------
    this.is_connected = function() {
        return fiwss_is_connected(this.fiwss_fcp_conn);
    };

    //---------------------------------------------------------------
    // Was the connected playback connection lost by remote WSS?
    //---------------------------------------------------------------
    this.lost_connection = function() {
        // Make sure we have a connection.
        if (this.fiwss_fcp_conn !== null) {
            // Did we lose the connection?
            if (this.fiwss_fcp_lost_connection) return true;
        }
        return false;
    };

    //---------------------------------------------------
    // Writes to the fcp stream
    // Return values:
    // 1 if the request was sent to WSS
    // 2 if the request was unable to be sent because it is waiting
    // -1 if the request was unable to be sent because there is no connection
    //---------------------------------------------------
    this.send_request = function(in_requestDataPkg, in_cbhCallMeWhenResponsed) {
        // Make sure we are connected.
        if (this.is_connected()) {
            // If already in a requesting state cannot continue.
            if (this.fiwss_fcp_requesting) return this.sendWaitingOnPastRequest;

            // Mark the callback handler for when the send completes
            // and the JSON request package is returned.
            this.cbhCallMeWhenResponsed = in_cbhCallMeWhenResponsed;

            // Save the request data package to install the response
            // data into ---
            this.requestDataPkg = in_requestDataPkg;

            // Send the request and block future requests until serviced
            this.fiwss_fcp_conn.send(in_requestDataPkg.jsonRequestData);
            this.fiwss_fcp_requesting = true;

            // All is well.
            return this.sendSentRequest;
        }

        // No connection available to transmit on.
        return this.sendNoConnection;
    };
}

//------------------------------------------------------------------------------
//******************************************************************************
//------------------------------------------------------------------------------
// FCP LISTENER SUPPORT
//------------------------------------------------------------------------------
//******************************************************************************
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// fiwss_fcp_listener is the object that manages the connection to Fastec Imaging
// Web Socket Server using the fi-fcp-listener-protocol identifier. The FCP
// listener protocol is used to read and write FCP registers to the camera.
//
// Expects a callback handler for connection and disconnection states.
//------------------------------------------------------------------------------
function fiwss_listener() {
    fiwss_listen_str = {
        fiwss_listener_connected:
            "FCP Listener Connection Established<br>Please Stand By...",
        fiwss_listener_noconnect:
            "Unable to Make FCP Listener Connection<br><br>Check FCP Listener on Camera<br><br>Try to Refresh Browser to Attempt New Connection.",
        fiwss_listener_disconnected:
            "Lost Connection to FCP Listener<br><br>Try to Refresh Browser to Reconnect."
    };

    // We need a reference back to "this" object. We use this in the body
    // of the event handlers defined below.
    var static_handle = this;

    // The JSON result value.
    this.jsonResult = null;

    // The connection socket - null when not connected.
    this.fiwss_listener_conn = null;

    // True if the WSS is connected and ready to send and receive listener requests.
    this.fiwss_listener_has_connection = false;

    // Keep a state of when a connection was lost (means it was opened
    // then closed by the remote WSS).
    this.fiwss_listener_lost_connection = false;

    // Using the default port of 7681
    this.fi_wssIPAddress = fiwss_make_url() + g_fiwss_default_port;

    // This name is used to declare the type of protocol on the WSS connection.
    this.listener_protocol_type = "fi-fcp-listener-protocol";

    // store the callback handler for when connection is completed
    this.cbhCallMeWhenConnected = null;

    // store the callback handler for when connection is disconnected
    this.cbhCallMeWhenDisconnected = null;

    // store the callback handler for when the json response package is
    // received on the wss socket
    this.cbhCallMeWhenResponsed = null;

    // keep a state of when a request is pending
    this.fiwss_listener_requesting = false;

    // keep a copy of the data package being sent on socket
    this.requestDataPkg = null;

    // code responses from send request
    this.sendNoConnection = -1;
    this.sendWaitingOnPastRequest = 2;
    this.sendSentRequest = 1;

    //---------------------------------------------------------------
    // Make a connection object to use for WSS fcp listener protocol
    //---------------------------------------------------------------
    this.create_wss_socket = function() {
        // If already connected cannot connect again
        if (this.fiwss_listener_conn !== null) return; // already has connect object don't create it again

        // Depending on the type of browser determines the type of socket to create
        if (typeof MozWebSocket != "undefined") {
            this.fiwss_listener_conn = new MozWebSocket(
                this.fi_wssIPAddress,
                this.listener_protocol_type
            );
        } else {
            this.fiwss_listener_conn = new WebSocket(
                this.fi_wssIPAddress,
                this.listener_protocol_type
            );
        }

        // These are the WSS handlers for web socket FCP listener interfaces used in browser.
        //
        // NOTE: We can't use "this" in the event handlers, because that won't be
        // 	 the correct "this". So we have our "static_handle" to get back to
        // 	 our real instance.
        try {
            //---------------------------------------------------------------
            // Callback function is called when FCP listener connection is established.
            //---------------------------------------------------------------
            this.fiwss_listener_conn.onopen = function() {
                static_handle.fiwss_listener_has_connection = true;
                gFastecAppMgr.vfInfoBoxShow(
                    fiwss_listen_str.fiwss_listener_connected,
                    false
                );

                // make a call to the cbh for connection status notification
                if (static_handle.cbhCallMeWhenConnected !== null)
                    static_handle.cbhCallMeWhenConnected();
            };

            //---------------------------------------------------------------
            // callback function is called when FCP listener connection established
            //---------------------------------------------------------------
            this.fiwss_listener_conn.onopen = function() {
                static_handle.fiwss_listener_has_connection = true;
                gFastecAppMgr.vfInfoBoxShow(
                    fiwss_listen_str.fiwss_listener_connected,
                    false
                );

                // Make a call to the callback handler for connection status notification.
                if (static_handle.cbhCallMeWhenConnected !== null)
                    static_handle.cbhCallMeWhenConnected();
            };

            //---------------------------------------------------------------
            // callback function is called when listener message is received from WSS
            //---------------------------------------------------------------
            this.fiwss_listener_conn.onmessage = function got_packet(msg) {
                if (static_handle.requestDataPkg !== null) {
                    // the parse should pull out the object data into the
                    // request data package in usable form
                    static_handle.requestDataPkg.jsonResponseData = JSON.parse(
                        msg.data,
                        function(key, value) {
                            var type;
                            if (value && typeof value === "object") {
                                type = value.type;
                                if (
                                    typeof type === "string" &&
                                    typeof window[type] === "function"
                                ) {
                                    return new window[type](value);
                                }
                            }
                            return value;
                        }
                    );

                    // Make sure to note that this request has been serviced.
                    static_handle.fiwss_listener_requesting = false;

                    // Call the callback handler.
                    if (static_handle.cbhCallMeWhenResponsed !== null)
                        static_handle.cbhCallMeWhenResponsed(
                            static_handle.requestDataPkg
                        );
                }
            };

            //---------------------------------------------------------------
            // this function is called when playback connection disconnected
            //---------------------------------------------------------------
            this.fiwss_listener_conn.onclose = function() {
                // Did we ever have a connection?
                if (static_handle.fiwss_listener_has_connection)
                    static_handle.fiwss_listener_lost_connection = true; // we lost a connected WSS
                else static_handle.fiwss_listener_lost_connection = false; // we never had an open connection

                static_handle.fiwss_listener_has_connection = false;
                static_handle.fiwss_listener_conn = null;

                // TODO - what if waiting on FCP command when wss disconnects?
                // it shouldn't matter.  but can get in a state where stuck?

                if (static_handle.fiwss_listener_lost_connection)
                    gFastecAppMgr.vfInfoBoxShow(
                        fiwss_listen_str.fiwss_listener_disconnected,
                        false
                    );
                else
                    gFastecAppMgr.vfInfoBoxShow(
                        fiwss_listen_str.fiwss_listener_noconnect,
                        false
                    );

                // Invoke callback handler to notify WSS is disconnected.
                if (static_handle.cbhCallMeWhenDisconnected !== null)
                    static_handle.cbhCallMeWhenDisconnected();
            };
        } catch (exception) {
            alert("FCP Listener Error: " + exception);
        }
    };

    //---------------------------------------------------------------
    // Attempt to reconnect with the WSS fcp listener protocol
    //---------------------------------------------------------------
    this.reconnect = function() {
        if (this.fiwss_listener_lost_connection) {
            // the wss socket was closed and must be recreated
            this.create_wss_socket();

            // If not able to make connection object report the problem.
            if (this.fiwss_listener_conn === null)
                gFastecAppMgr.vfInfoBoxShow(
                    fiwss_listen_str.fiwss_listener_noconnect,
                    false
                );
        }
    };

    //---------------------------------------------------------------
    // Is the fcp listener ready for communications?
    //---------------------------------------------------------------
    this.is_connected = function() {
        return fiwss_is_connected(this.fiwss_listener_conn);
    };

    //---------------------------------------------------------------
    // Was the connected playback connection lost by remote WSS?
    //---------------------------------------------------------------
    this.lost_connection = function() {
        // make the connection to WSS for liveview communications
        if (this.fiwss_listener_conn !== null) {
            if (this.fiwss_listener_lost_connection) return true;
        }
        return false;
    };

    //---------------------------------------------------
    // Writes to the fcp listener
    // Return values:
    // 1 if the request was sent to WSS
    // 2 if the request was unable to be sent because it is waiting
    // -1 if the request was unable to be sent because there is no connection
    //---------------------------------------------------
    this.send_request = function(in_requestDataPkg, in_cbhCallMeWhenResponsed) {
        // Make sure we're connected.
        if (this.is_connected()) {
            // If already in a requesting state cannot continue
            if (this.fiwss_listener_requesting)
                return this.sendWaitingOnPastRequest;

            // Mark the callback handler for when the send completes
            // and the JSON request package is returned.
            this.cbhCallMeWhenResponsed = in_cbhCallMeWhenResponsed;

            // Save the request data package to install the response
            // data into ---
            this.requestDataPkg = in_requestDataPkg;

            // Send the request and block future requests until serviced
            this.fiwss_listener_conn.send(in_requestDataPkg.jsonRequestData);
            this.fiwss_listener_requesting = true;

            // All done.
            return this.sendSentRequest;
        }

        // No connection available to transmit on.
        return this.sendNoConnection;
    };
}
