//------------------------------------------------
// fastec_toolbar.js
//
// Copyright 2010-2018 (c) Fastec Imaging as an unpublished work.
// All Rights Reserved.
//
// The information contained herein is confidential property of
// Fastec Imaging. The use, copying, transfer or disclosure of such
// information is prohibited except by express written agreement
// with Fastec Imaging.
//
// Fastec Imaging Camera Control Web Application
// Toolbar Object
//------------------------------------------------

//------------------------------------
// The Fastec Toolbar Button Object
//
// This object is responsible for managing
// the state of a toolbar button.
//------------------------------------
function fastecToolbarBtn() {
    try {
        // Save the JQuery Selector ID for this toolbar button.
        this.theTBSelector = 0;

        // Save the attribute id for fast comparisons.
        this.theAttrID = " ";

        // Default the toolbar button as active	but can be turned off
        // so that nothing happens when events occur with the toolbar button.
        this.toolbarActive = true;

        // Is this toolbar button visible?
        this.toolbarVisible = true;

        // A click handler for this toolbar button.
        this.clickHandlerFunc = null;

        // Stores as bound data to our object instance
        this.idData = { fastecToolbarBtn: this };

        //---------------------------
        //---------------------------
        this.onTBClick = function(e) {
            if (e.data.fastecToolbarBtn != undefined) {
                var me = e.data.fastecToolbarBtn;

                fastecLogToConsole(gLogAll, "onTBClick() - id=" + me.theAttrID);

                // Is the toolbar active and do we have a click handler?
                if (me.toolbarActive && me.clickHandlerFunc !== null) {
                    // Call the click handler.
                    me.clickHandlerFunc();
                }
            }
        };

        //---------------------------
        //---------------------------
        this.onTBMouseOver = function(e) {
            if (e.data.fastecToolbarBtn != undefined) {
                var me = e.data.fastecToolbarBtn;

                fastecLogToConsole(
                    gLogAll,
                    "onTBMouseOver() - id=" + me.theAttrID
                );

                if (me.toolbarActive) {
                    //me.theTBSelector.removeClass('tbUnSelect');
                    me.theTBSelector.addClass("tbSelect");
                }
            }
        };

        //---------------------------
        //---------------------------
        this.onTBMouseOut = function(e) {
            if (e.data.fastecToolbarBtn != undefined) {
                var me = e.data.fastecToolbarBtn;

                fastecLogToConsole(
                    gLogAll,
                    "onTBMouseOut() - id=" + me.theAttrID
                );

                if (me.toolbarActive) {
                    me.theTBSelector.removeClass("tbSelect");
                }
            }
        };

        //---------------------------
        //---------------------------
        this.onTBMouseDown = function(e) {
            if (e.data.fastecToolbarBtn != undefined) {
                var me = e.data.fastecToolbarBtn;

                fastecLogToConsole(
                    gLogAll,
                    "onTBMouseDown() - id=" + me.theAttrID
                );

                if (me.toolbarActive) me.theTBSelector.fadeTo(0, 0.55);
            }
        };

        //---------------------------
        //---------------------------
        this.onTBMouseUp = function(e) {
            if (e.data.fastecToolbarBtn != undefined) {
                var me = e.data.fastecToolbarBtn;

                fastecLogToConsole(
                    gLogAll,
                    "onTBMouseUp() - id=" + me.theAttrID
                );

                if (me.toolbarActive) me.theTBSelector.fadeTo(0, 1);
            }
        };

        //---------------------------
        // Set the JQuery Selector
        //---------------------------
        this.setTBSelector = function(inTBSelector) {
            this.theTBSelector = inTBSelector;
            this.theAttrID = this.theTBSelector.attr("id");
            this.theTBSelector.bind("click", this.idData, this.onTBClick);
            this.theTBSelector.bind(
                "mouseover",
                this.idData,
                this.onTBMouseOver
            );
            this.theTBSelector.bind("mouseout", this.idData, this.onTBMouseOut);
            this.theTBSelector.bind(
                "mousedown",
                this.idData,
                this.onTBMouseDown
            );
            this.theTBSelector.bind("mouseup", this.idData, this.onTBMouseUp);
        };

        //---------------------------
        // Set the button active state
        //---------------------------
        this.setActiveState = function(amIActive) {
            if (amIActive) {
                this.toolbarActive = true;
                this.theTBSelector.fadeTo(0, 1);
            } else {
                this.toolbarActive = false;
                this.theTBSelector.fadeTo(0, 0.33);
            }
        };

        //---------------------------
        // Set the button visible state
        //---------------------------
        this.setVisibleState = function(amIVisible) {
            if (amIVisible) {
                this.toolbarVisible = true;
                this.theTBSelector.show();
            } else {
                this.toolbarActive = false;
                this.theTBSelector.hide();
            }
        };

        //---------------------------
        // Set the click handler for this button
        //---------------------------
        this.setClickHandler = function(theClickHandlerFunc) {
            this.clickHandlerFunc = theClickHandlerFunc;
            this.clickHandlerExists = true;
        };
    } catch (err) {
        var theDetails = "caught: fastecToolbarBtn -" + err.description;
        gFaultHandler.logError(theDetails);
    }
}

//------------------------------------
// The Fastec Toolbar Manager Object
//
// This object is responsible for managing
// the state of the main toolbar.
//------------------------------------
function fastecToolbarMgr() {
    tbmgr_str = {
        armTBTitle: "Click To ARM Camera",
        armTBTxtTitle: "Arm",
        dearmTBTitle: "Click To Stop Capture",
        dearmTBTxtTitle: "Armed",
        trigTBTitle: "Click To Trigger Camera",
        detrigTBTitle: "Click To Turn Off Trigger State On Camera",
        liveTBTitle: "Click to Go Live View on Camera",
        reviewTBTitle: "Click to Review Captured Partition"
    };

    try {
        // each of the toolbar buttons will have a unique object managed
        // together by this container object
        this.armTB = new fastecToolbarBtn();
        this.triggerTB = new fastecToolbarBtn();
        this.liveTB = new fastecToolbarBtn();
        this.revTB = new fastecToolbarBtn();
        this.liveViewToggleTB = new fastecToolbarBtn();

        //---------------------------
        // Toolbar initialization
        //---------------------------
        this.initFastecToolbar = function() {
            this.armTB.setTBSelector($("#armTBID"));
            this.triggerTB.setTBSelector($("#triggerTBID"));
            this.liveTB.setTBSelector($("#liveTBID"));
            this.revTB.setTBSelector($("#reviewTBID"));
            this.liveViewToggleTB.setTBSelector($("#liveViewTogTBID"));

            // initialize states of buttons to be inactive
            this.armTB.setActiveState(false);
            this.armTB.setVisibleState(true);
            this.armTB.setClickHandler(gFastecAppMgr.onArmTBClick);

            this.triggerTB.setActiveState(false);
            this.triggerTB.setVisibleState(true);
            this.triggerTB.setClickHandler(gFastecAppMgr.onTriggerTBClick);

            this.liveTB.setActiveState(false);
            this.liveTB.setVisibleState(true);
            this.liveTB.setClickHandler(gFastecAppMgr.onLiveTBClick);
            $("#liveTBID").attr("title", tbmgr_str.liveTBTitle);

            this.revTB.setActiveState(false);
            this.revTB.setVisibleState(true);
            this.revTB.setClickHandler(gFastecAppMgr.onReviewTBClick);
            $("#revTBID").attr("title", tbmgr_str.reviewTBTitle);

            this.liveViewToggleTB.setActiveState(true);
            this.liveViewToggleTB.setVisibleState(true);
            this.liveViewToggleTB.setClickHandler(
                gFastecAppMgr.onLiveViewToggleTB
            );
        };

        //---------------------------
        // common handler to setup
        // titles and images for arm/trigger buttons
        //---------------------------
        this.setArmTrigTBTitles = function(iAmArmed, okToTrigger, okToArm) {
            fastecLogToConsole(
                gLogTrace,
                "setArmTrigTBTitles(" +
                    iAmArmed +
                    ", " +
                    okToTrigger +
                    ", " +
                    okToArm +
                    ")"
            );

            if (iAmArmed) {
                $("#armTBID")
                    .removeClass("ui-icon-arm")
                    .addClass("ui-icon-dearm");
                $("#armTBID").attr("title", tbmgr_str.dearmTBTitle);
                $("#triggerTBID").attr("title", tbmgr_str.trigTBTitle);
                $("#armTBTextID").html(tbmgr_str.dearmTBTxtTitle);
                gToolbarMgr.armTB.setActiveState(okToArm);
                gToolbarMgr.armTB.setVisibleState(true);
                gToolbarMgr.triggerTB.setActiveState(okToTrigger);
                gToolbarMgr.triggerTB.setVisibleState(true);
            } else {
                $("#armTBID")
                    .removeClass("ui-icon-dearm")
                    .addClass("ui-icon-arm");
                $("#armTBID").attr("title", tbmgr_str.armTBTitle);
                $("#armTBTextID").html(tbmgr_str.armTBTxtTitle);
                gToolbarMgr.armTB.setActiveState(okToArm);
                gToolbarMgr.armTB.setVisibleState(true);
                gToolbarMgr.triggerTB.setActiveState(false);
                gToolbarMgr.triggerTB.setVisibleState(true);
            }
        };

        //---------------------------
        // common handler to setup
        // titles and images for live view toggle button
        //---------------------------
        this.setLiveViewToggleTBState = function(activeTB, usePlayButton) {
            fastecLogToConsole(
                gLogTrace,
                "setLiveViewToggleTBState(" +
                    activeTB +
                    ", " +
                    usePlayButton +
                    ")"
            );

            // Set up the appropriate icon in toolbar button
            if (!usePlayButton) {
                $("#liveViewTogTBID")
                    .removeClass("ui-icon-play-forward")
                    .addClass("ui-icon-pause");
                $("#liveViewTogTBID").attr(
                    "title",
                    tbmgr_str.liveViewPauseTBTitle
                );
            } else {
                $("#liveViewTogTBID")
                    .removeClass("ui-icon-pause")
                    .addClass("ui-icon-play-forward");
                $("#liveViewTogTBID").attr(
                    "title",
                    tbmgr_str.liveViewPlayTBTitle
                );
            }
            gToolbarMgr.liveViewToggleTB.setActiveState(activeTB);
        };
    } catch (err) {
        var theDetails = "caught: fastecToolbarMgr-" + err.description;
        gFaultHandler.logError(theDetails);
    }
}

