/*!
 * @file fastec_sensor_lince.cpp
 * @brief This is the javascript object responsible for housing
 *        Lince-specific sensor related record setting calculations
 *
 * Copyright 2014 (c) Fastec Imaging as an unpublished work.
 * All Rights Reserved.
 *
 * The information contained herein is confidential property of
 * Fastec Imaging. The use, copying, transfer or disclosure of such
 * information is prohibited except by express written agreement
 * with Fastec Imaging.
 */

/*!
 * @brief This is the header file providing the hardware abstraction layer
 * for the Lince5M and its sensor-specific FPGA interface.
 *
 * Copyright 2010, 2014 (c) Fastec Imaging as an unpublished work.
 * All Rights Reserved.
 */
var PIXELS_PER_PPW = 16; //!< Sensor groups pixels into PPWs "pixel processing windows".
var PPWS_PER_BLOCK = 4; //!< Sensor emits multiples of 64 pixels, which is 4 PPWs.

//! First line of defect map file.
var LINCE5M_DEFECT_HEADER = "AL5Xv1:%u\n";

//! Maximum number of pixel defects allowed. Anafocus manufacturing permits up
//! to 100 whitespot pixels and up to 150 black/bright pixels.
var MAX_DEFECTS = 250;

//! The sensor emits pixels in groups of PPWs and wants the number of PPWs to be
//! multiple of 4, so we tell the FPGA how to extract pixels from these blocks of 64.
var PIXELS_PER_BLOCK = PIXELS_PER_PPW * PPWS_PER_BLOCK;

//! Lince5M sensor actual size.
var LINCE_SENSOR_WIDTH = 2560;
var LINCE_SENSOR_HEIGHT = 2048;

//! The maximum size image we allow, what we expose to the user.
//! We aren't ready to implement the full sensor size yet so this is what we report.
var LINCE_REPORTED_MAX_WIDTH = LINCE_SENSOR_WIDTH;
var LINCE_REPORTED_MAX_HEIGHT = LINCE_SENSOR_HEIGHT;

// Maximum ROI we permit in 10-bit mode
var LINCE_MAX_WIDTH_10BIT = LINCE_SENSOR_WIDTH;
var LINCE_MAX_HEIGHT_10BIT = LINCE_SENSOR_HEIGHT;

// Maximum ROI we permit in 12-bit mode
var LINCE_MAX_WIDTH_12BIT = LINCE_SENSOR_WIDTH;
var LINCE_MAX_HEIGHT_12BIT = LINCE_SENSOR_HEIGHT;

//! Minimum ROI we are supporting currently.
var LINCE_ROI_MIN_WIDTH = 128;
var LINCE_ROI_MIN_HEIGHT = 64;

//! Default frame rate when none has been set.
var LINCE_DEF_FRAME_RATE = 250;

//! Maximum width to use register files with faster row timing.
var TROW_52_MAX_WIDTH = 768;
var TROW_84_MAX_WIDTH = 1280;
var TROW_124_MAX_WIDTH = 1920;
var TROW_132_MAX_WIDTH = 2048;

//! Values from Lince5M_Regs_FM1_86.4MHz_trow{52,84}_HSHS[_bin2x2]_rev910.dat
//! The values for [_bin2x2] for trow=84 are the same as the normal mode file.
var FM1_N_CYCLES_B_52 = 0x000000cb; //! Rg0x004-Rg0x005 for t_row=52. (203)
var FM1_N_CYCLES_F_52 = 0x000002a8; //! Rg0x006-Rg0x007 for t_row=52. (680)
var FM1_N_CYCLES_B_84 = 0x00000075; //! Rg0x004-Rg0x005 for t_row=84. (117)
var FM1_N_CYCLES_F_84 = 0x000000fb; //! Rg0x006-Rg0x007 for t_row=84. (251)
//! Values from Lince5M_Regs_FM1_86.4MHz_trow{124,132,165}_HSHS_rev707.dat
var FM1_N_CYCLES_B_124 = 0x000000cb; //! Rg0x004-Rg0x005 for t_row=124. (203)
var FM1_N_CYCLES_F_124 = 0x000003fe; //! Rg0x006-Rg0x007 for t_row=124. (1022)
var FM1_N_CYCLES_B_132 = 0x000000cb; //! Rg0x004-Rg0x005 for t_row=132. (203)
var FM1_N_CYCLES_F_132 = 0x000003fe; //! Rg0x006-Rg0x007 for t_row=132. (1022)
var FM1_N_CYCLES_B_165 = 0x000000cb; //! Rg0x004-Rg0x005 for t_row=165. (203)
var FM1_N_CYCLES_F_165 = 0x0000035a; //! Rg0x006-Rg0x007 for t_row=165. (858)
//! Values from Lince5M_Regs_FM6_72MHz_trow84_HSHS_rev914.dat
//! The values for [_bin2x2] for trow=84 are the same as the normal mode file.
var FM6_N_CYCLES_B_84 = 0x00000062; //! Rg0x004-Rg0x005 for t_row=84. (98)
var FM6_N_CYCLES_F_84 = 0x000000d3; //! Rg0x006-Rg0x007 for t_row=84. (211)
//! Values from Lince5M_Regs_FM6_72MHz_trow{124,132,165}_HSHS_rev{559,558,622}.dat
var FM6_N_CYCLES_B_124 = 0x000000a8; //! Rg0x004-Rg0x005 for t_row=124. (168)
var FM6_N_CYCLES_F_124 = 0x000002df; //! Rg0x006-Rg0x007 for t_row=124. (735)
var FM6_N_CYCLES_B_132 = 0x000000a8; //! Rg0x004-Rg0x005 for t_row=132. (168)
var FM6_N_CYCLES_F_132 = 0x000002df; //! Rg0x006-Rg0x007 for t_row=132. (735)
var FM6_N_CYCLES_B_165 = 0x000000a8; //! Rg0x004-Rg0x005 for t_row=165. (168)
var FM6_N_CYCLES_F_165 = 0x000002d3; //! Rg0x006-Rg0x007 for t_row=165. (723)

var EXTRA_ROWS = 2; //! Rg0x00A, the same for all files in both framing modes.
var EXTRA_CYCLES = 1; //! Rg0x00b-Rg0x00c, the same for all files in both framing modes.

// Even if these are not needed for the common utilities, include them here just
// so all the values we have to extract from the register files are together.
//
// Values from Lince5M_Regs_FM1_86.4MHz_trow{52,84}_HSHS[_bin2x2]_rev910.dat
// The values for [_bin2x2] for trow=84 are the same as the normal mode file.
var FM1_ANALOG_GAIN_52 = 0x001e; //! Rg0x05B for t_row=52.
var FM1_ANALOG_GAIN_84 = 0x001e; //! Rg0x05B for t_row=84.
// Values from Lince5M_Regs_FM1_86.4MHz_trow{124,132,165}_HSHS_rev707.dat
var FM1_ANALOG_GAIN_124 = 0x0020; //! Rg0x05B for t_row=124.
var FM1_ANALOG_GAIN_132 = 0x0020; //! Rg0x05B for t_row=132.
var FM1_ANALOG_GAIN_165 = 0x001e; //! Rg0x05B for t_row=165.
// Values from Lince5M_Regs_FM6_72MHz_trow84_HSHS[_bin2x2]_rev910.dat
// The values for [_bin2x2] for trow=84 are the same as the normal mode file.
var FM6_ANALOG_GAIN_84 = 0x001e; //! Rg0x05B for t_row=84.
// Values from Lince5M_Regs_FM6_72MHz_trow{124,132,165}_HSHS_rev{559,558,622}.dat
var FM6_ANALOG_GAIN_124 = 0x0020; //! Rg0x05B for t_row=124.
var FM6_ANALOG_GAIN_132 = 0x0020; //! Rg0x05B for t_row=132.
var FM6_ANALOG_GAIN_165 = 0x001d; //! Rg0x05B for t_row=165.

//! Lince Frame Overhead Time is the duration of State F.
//! For FM1, max value of N_CYCLES_F in Rg0x006-Rg0x007 is 1022. At 86.4MHz, that's 11.8 usecs.
//! For FM6, max value of N_CYCLES_F in Rg0x006-Rg0x007 is 735. At 72MHz, that's 10.2 usecs.
//! We take the worst case and use 12 usecs. Using too large a value means that maybe
//! we don't eke out one more frame per second in some case.
var LINCE_FOT_USEC = 12;

//! Lince minimum exposure time is the duration of State B which is no exposure at all.
//! For FM1, max value of N_CYCLES_B in Rg0x004-Rg0x005 is 203. At 86.4MHz, that's 2.35 usecs.
//! For FM6, max value of N_CYCLES_B in Rg0x004-Rg0x005 is 168. At 72MHz, that's 2.33 usecs.
//! But for trow=52 where we don't have separate files for FM1 vs FM6, N_CYCLES_B is 203.
//! At 72Mhz, that's 2.82 usecs -- lucky break, still under 3 usecs!
//! We can just use the same value for all framing modes.
var LINCE_MIN_EXP_USEC = 3;

//! f_sys is the system frequency, which depends on the framing mode.
var FM1_F_SYS = 86400000; // 86.4 MHz for Framing Mode 1
var FM6_F_SYS = 72000000; // 72.0 MHz for Framing Mode 6

/*
 * Rg0x000 = trigger configuration
 * Rg0x001 = device status
 * Rg0x002-Rg0x003 = integration time
 * Rg0x004-Rg0x005 = N.cycles_B, duration of State B.
 * Rg0x006-Rg0x007 = N.cycles_F, duration of State F.
 * Rg0x00a = extra rows to adjust frame rate by adding vertical blanking
 * Rg0x00b-Rg0x00c = extra cycles to add to readout to adjust frame rate
 * Rg0x010-0x013 = ROI (row, column, num rows, num cols)
 * Rg0x014 = subsampling and binning
 * Rg0x015 = t.row_min_cycles, t.row_cycles, can increase to slow frame rate
 * Rg0x017-Rg0x20 = Global digital gains and offsets
 * Rg0x01c = used in VFPN calibration
 * Rg0x024 = LVDS interface configuration
 * Rg0x025-Rg0x026, Rg0x036, Rg0x038 - bits per pixel and LVDS pad configuration
 * Rg0x027-Rg0x033 = LVDS port skews
 * Rg0x034 = IDLE pattern
 * Rg0x036 = coarse delays for LVDS ports
 * Rg0x038,Rg0x039 = framing mode configuration and IDLE pattern for CMOS
 * Rg0x048-Rg0x049 = used for calibration in "strong light conditions" and for HDR
 * Rg0x04a = used for HDR configuration
 * Rg0x04b = either 0x1740 or 0x174c depending on whether sensor is covered in calibration
 * Rg0x04c = used for HDR configuration
 * Rg0x04d = used for calibration in HDR mode
 * Rg0x05b = analog gain
 * Rg0x065 = analog offset
 * Rg0x06a = used for HDR configuration
 * Rg0x06b = used for HDR configuration
 * Rg0x070 = used for HDR configuration
 * Rg0x07c = polarization current of the pixel, increased for binning with stressed trow
 * Rg0x085 = used for HDR configuration
 * Rg0x087-Rg0x091 = modify from defaults for HDR configuration (3.4.2.1)
 * Rg0x0e8-Rg0x0e9 = used for calibration in "strong light conditions"
 * Rg0x14d-Rg0x14e = for adding horizontal blanking
 * Rg0x151-Rg0x152 = component of fixed latency between trigger start and exposure start
 * Rg0x153-Rg0x154 = component of fixed latency between trigger start and exposure start
 * Rg0x155-Rg0x156 = component of fixed latency between trigger start and exposure start
 * Rg0x157-Rg0x158 = component of minimum time trigger must remain inactive
 * Rg0x159-Rg0x15a = component of minimum time trigger must remain inactive
 * Rg0x15b-Rg0x15c = component of minimum time trigger must remain inactive
 * Rg0x15d-Rg0x15e = component of minimum time trigger must remain inactive
 * Rg0x15f-Rg0x160 = component of minimum time trigger must remain inactive
 * Rg0x168 = modify from defaults for HDR configuration (3.4.2.1)
 * Rg0x170-Rg0x171 = control programmable memories access
 * Rg0x172 = constant latency, minimum cycles between trigger starts + 1 (single-edge mode)
 * Rg0x174 = used for calibration in HDR mode
 * Rg0x176, Rg0x180 = to support Hot ROI
 * Rg0x17e-Rg0x17f = disable groups of optically black pixels, e.g. if defective
 * Rg0x180 = cfg_all_array, force enable all active array even if outside ROI
 * Rg0x1a9 = compensation digital offset
 * Rg0x1aa, Rg0x1ab = something about VFPN offset
 * Rg0x1ad = unlock VFPN offset correction memories (0x0a00-0x13ff)
 * Rg0x1ae = control defective pixel correction
 * Rg0x1b5 = has enable for HFPN offset correction
 * Rg0x1b7 = enable global digital gains and offsets
 * Rg0x1bc = enable statistics module of DIP
 * Rg0x1cb-Rg0x1cc = read OB column sum
 */

//! Convert integration in microseconds to sys clock counts
//! In Framing Mode 1, f.sys = 86.4 MHz.
function FM1_INTEGRATION_USECS_TO_FSYS(u) {
    var result = (u * 864 + 5) / 10;
    return result;
}

//! In Framing Mode 6, f.sys = 72.0 MHz.
function FM6_INTEGRATION_USECS_TO_FSYS(u) {
    var result = (u * 720 + 5) / 10;
    return result;
}

/*!
 * @brief Convert user ROI to sensor ROI
 *
 * Convert user ROI coordinates to sensor coordinates. Generally this is a matter of multiplying
 * by the binning/subsampling factor, but if the factor is 1 we apply an offset so the user
 * is always working with the center 1920x1080 (8- or 10-bit) or 1280x1024 (12-bit).
 *
 * @param factor	The binning/subsampling factor.
 * @param startx	Starting x offset in the user's ROI.
 * @param starty	Starting y offset in the user's ROI.
 * @param width		The width of the user's ROI.
 * @param height	The height of the user's ROI.
 * @param bit_mode	The bit mode to use, which can affect the interpretation of the ROI.
 * @param this.startx	sets the starting x offset in sensor coordinates.
 * @param this.starty	sets the starting y offset in sensor coordinates.
 * @param this.width	sets the width in sensor coordinates.
 * @param this.height	sets the height in sensor coordinates.
 *
 * @return lince5m_user_to_sensor_result object which has any_changes set to
 * nonzero if we had to adjust the ROI to make it legal, zero if all is okay.
 */
//--------------------------------------------------------------------
// the result object for function lince5m_user_to_sensor_result
//--------------------------------------------------------------------
function lince5m_user_to_sensor_result() {
    // object to hold ROI values in one place
    this.sensor_x = 0;
    this.sensor_y = 0;
    this.sensor_width = 0;
    this.sensor_height = 0;
    // any changes will store a state of change to data from
    // the original calculations
    this.any_changes = 0;
}
function lince5m_user_to_sensor(
    factor,
    startx,
    starty,
    width,
    height,
    bit_mode
) {
    var any_changes = 0;

    // make a return ROI object.
    var rtn_ROI = new lince5m_user_to_sensor_result();

    // build local variables to use in this code
    var rtn_startx = startx * factor;
    var rtn_starty = starty * factor;
    var rtn_width = width * factor;
    var rtn_height = height * factor;

    // Now center the user-visible region within what we allow.
    // With factor=1, we only show the user either the center 1920x1080 region or the
    // center 1280x1024 region, so the user's ROI offset is within that area.
    // With factor >1, the entire sensor area is available so the offsets are already okay.
    if (factor == 1) {
        var offset;

        // create the javascript object that can hold multiple return values
        var rtn_MaxROI = lince5m_max_roi_setting(bit_mode);

        offset = (LINCE_SENSOR_WIDTH - rtn_MaxROI.max_width) / 2;
        if (rtn_startx + rtn_width + offset <= LINCE_SENSOR_WIDTH)
            rtn_startx += offset;

        offset = (LINCE_SENSOR_HEIGHT - rtn_MaxROI.max_height) / 2;
        if (rtn_starty + rtn_height + offset <= LINCE_SENSOR_HEIGHT)
            rtn_starty += offset;
    }

    if (rtn_width > LINCE_SENSOR_WIDTH) {
        rtn_width = LINCE_SENSOR_WIDTH;
        any_changes = 1;
    }
    if (rtn_height > LINCE_SENSOR_HEIGHT) {
        rtn_height = LINCE_SENSOR_HEIGHT;
        any_changes = 1;
    }
    if (rtn_startx + rtn_width > LINCE_SENSOR_WIDTH) {
        rtn_startx = LINCE_SENSOR_WIDTH - rtn_width;
        any_changes = 1;
    }
    if (rtn_starty + rtn_height > LINCE_SENSOR_HEIGHT) {
        rtn_starty = LINCE_SENSOR_HEIGHT - rtn_height;
        any_changes = 1;
    }

    // return the data structure created by these calculations
    rtn_ROI.any_changes = any_changes;
    rtn_ROI.sensor_x = rtn_startx;
    rtn_ROI.sensor_y = rtn_starty;
    rtn_ROI.sensor_width = rtn_width;
    rtn_ROI.sensor_height = rtn_height;

    return rtn_ROI;
}

/*!
 * @brief Get the readout dims we need to give the sensor to satisfy boundary constraints
 *
 * Sensor readout is always in Pixel Processing Windows, 16 pixels, but the column window
 * we set up in the sensor must be at multiples of 4 PPWs or 64 pixels. Then if we are
 * binning or subsampling, the readout is guaranteed to be in multiples of 16 PPWs, since
 * the worst-case binning factor is 4.
 *
 * @param factor	The binning/subsampling factor
 * @param sensor_startx	The first sensor pixel in the user's ROI, in sensor coordinates
 * @param sensor_width	The width of the user's ROI, in sensor coordinates
 * @param rtn_startx	Return the first column to read in sensor coordinates
 * @param rtn_width	Return the number of column to read in sensor coordinates
 *
 * @return nonzero if we had to adjust the ROI to make it legal, zero if all is okay.
 */
//--------------------------------------------------------------------
// the result object for function lince5m_user_to_sensor_result
//--------------------------------------------------------------------
function lince5m_get_readout_dims_result() {
    // object to hold ROI values in one place
    this.readout_x = 0;
    this.readout_width = 0;
    this.any_changes = 0;
}
function lince5m_get_readout_dims(factor, sensor_startx, sensor_width) {
    var any_changes = 0;
    var readout_startx;
    var readout_endx;

    // Now we need to make sure the readout for each row is a multiple of 16 pixels even
    // after it has been subsampled. The startx we give the sensor must be a multiple of 64
    // to accommodate sensor restrictions.

    // This is the next lower multiple of 64 below startx.
    readout_startx = sensor_startx / PIXELS_PER_BLOCK * PIXELS_PER_BLOCK;
    readout_startx = fastecMakeInteger(readout_startx);

    // This is the next multiple of 64 higher than the rightmost pixel.
    //readout_endx = (( sensor_startx + sensor_width + PIXELS_PER_BLOCK - 1 ) / PIXELS_PER_BLOCK ) *
    //	    PIXELS_PER_BLOCK;
    // 03-17-15
    // must pull out the original logic and maintain unsigned integer
    // simulation because javascript will treat as floating point causing problems
    var keepIntegerBased =
        (sensor_startx + sensor_width + PIXELS_PER_BLOCK - 1) /
        PIXELS_PER_BLOCK;
    keepIntegerBased = fastecMakeInteger(keepIntegerBased);
    readout_endx = keepIntegerBased * PIXELS_PER_BLOCK;

    // When we readout, the FPGA wants a multiple of 16 pixels and there must be at least 64 pixels
    // for the arithmetic in last crop area. So as long as we have 5 PPWs left after binning,
    // we should be okay.
    var factorPPW = factor * (5 * PIXELS_PER_PPW); // create integer-based value
    factorPPW = fastecMakeInteger(factorPPW);

    if (readout_endx - readout_startx < factorPPW) {
        if (readout_startx >= PIXELS_PER_BLOCK)
            readout_startx -= PIXELS_PER_BLOCK;
        if (readout_endx - readout_startx < factorPPW) {
            if (readout_endx < LINCE_SENSOR_WIDTH - PIXELS_PER_BLOCK)
                readout_endx += PIXELS_PER_BLOCK;
        }
        if (readout_endx - readout_startx < factorPPW) any_changes = 1;
    }

    // build return object
    var rtnObj = new lince5m_get_readout_dims_result();
    rtnObj.readout_x = readout_startx;
    rtnObj.readout_width = readout_endx - readout_startx;
    rtnObj.any_changes = any_changes;

    return rtnObj;
}

/*!
 * @brief Compute the maximum frame rate for a given ROI.
 *
 * Frame time is the maximum of t_exp and t_readout.
 *
 * The value for t_exp includes the sensor states B, E, and F (t_exp_B, t_exp_E, t_exp_F),
 * where t_exp_E is the exposure time, t_exp_B is N_cycles_B / f_sys, and
 * t_exp_F = N_cycles_F / fsys).
 *
 * t_exp = t_exp_B + t_exp_E + t_exp_F
 *
 * The values for N_cycles_B and N_cycles_F come from the configuration, but we need those
 * values before the sensor is configured, so we just pre-define them based on what was in
 * those .dat files. There are separate configuration files for the three t_row values we support.
 *
 * t_readout = t_row_cycles * rows + extra-cycles + t_wait_cycles + N_cycles_F.
 *
 * @param readout_width	    The number of columns being read out from the sensor.
 * @param readout_height    The number of rows being read out from the sensor.
 *
 * @return The maximum number of frames per second
 */
function lince5m_get_max_frame_rate(readout_width, readout_height, sensopts) {
    var rows;
    var t_row_cycles;
    var t_readout;
    var t_wait_cycles;
    var N_cycles_B, N_cycles_F;
    var K;
    var fps;
    var f_sys;
    var use_fm6;

    // important: Javascript doesn't do high bit manipulation well - specifically in the
    // 0x80000000 case because it treats the high bit as a sign bit - the bit being looked
    // at here should not cause a sign bit error but what if in the future they change it
    // or add a high bit setting.  For now pull it out as a small byte and test against a byte
    // value -- this should work for all cases going forward...
    // #define GIGE_SENSOPT_INTERNAL_0    0x10000000	// Sensor-specific internal setting.
    // pull the most significant byte out (3-0) where 3 is most and 0 is least in GIGE register
    var sensopt_internal_0_byte = fastecHexToDec(csRegPullByte(sensopts, 3));
    var sensopt_internal_0 = false;
    if (sensopt_internal_0_byte & 0x10) sensopt_internal_0 = true;

    // test state of GIGE_SENSOPT_INTERNAL_0 bit
    if (sensopt_internal_0_byte) {
        use_fm6 = 1;
        f_sys = FM6_F_SYS;
    } else {
        use_fm6 = 0;
        f_sys = FM1_F_SYS;
    }

    // t_row_min depends on the loaded configuration, which depends on the readout width,
    // but we don't add any HBlanking_cycles. So t_row_cycles=t_row_min.
    //#define GIGE_SENSOPT_BIN_MASK      0x000000FF	// Binning option mask
    //#define GIGE_SENSOPT_BIN_SHIFT     0		// Anafocus binning is ( 1, 2, 4 ) << 0
    // pull the most least signifant byte out (3-0) where 3 is most and 0 is least in GIGE register
    var sensopt_bin_mask = fastecHexToDec(csRegPullByte(sensopts, 0));
    // added to match
    if (readout_width <= TROW_52_MAX_WIDTH && (sensopt_bin_mask & 0xff) > 1) {
        t_row_cycles = 52;
        N_cycles_B = FM1_N_CYCLES_B_52;
        N_cycles_F = FM1_N_CYCLES_F_52;
    } else if (readout_width <= TROW_84_MAX_WIDTH) {
        t_row_cycles = 84;
        N_cycles_B = use_fm6 ? FM6_N_CYCLES_B_84 : FM1_N_CYCLES_B_84;
        N_cycles_F = use_fm6 ? FM6_N_CYCLES_F_84 : FM1_N_CYCLES_F_84;
    } else if (readout_width <= TROW_124_MAX_WIDTH) {
        t_row_cycles = 124;
        N_cycles_B = use_fm6 ? FM6_N_CYCLES_B_124 : FM1_N_CYCLES_B_124;
        N_cycles_F = use_fm6 ? FM6_N_CYCLES_F_124 : FM1_N_CYCLES_F_124;
    } else if (readout_width <= TROW_132_MAX_WIDTH) {
        t_row_cycles = 132;
        N_cycles_B = use_fm6 ? FM6_N_CYCLES_B_132 : FM1_N_CYCLES_B_132;
        N_cycles_F = use_fm6 ? FM6_N_CYCLES_F_132 : FM1_N_CYCLES_F_132;
    } else {
        t_row_cycles = 165;
        N_cycles_B = use_fm6 ? FM6_N_CYCLES_B_165 : FM1_N_CYCLES_B_165;
        N_cycles_F = use_fm6 ? FM6_N_CYCLES_F_165 : FM1_N_CYCLES_F_165;
    }

    // (24) K is ( N_cycles_B / t_row_cycles ) rounded up plus 3.
    K = fastecMakeInteger((N_cycles_B + t_row_cycles - 1) / t_row_cycles + 3);
    rows = readout_height + EXTRA_ROWS + K;

    // (28) Extra time for internal synchronization.
    // The +1 is from the AnaFocus frame rate spreadsheet, which disagrees with the data sheet.
    t_wait_cycles = 3 * t_row_cycles + 1;

    // This is the readout time in f_sys cycles.
    t_readout = fastecMakeInteger(
        t_row_cycles * rows + EXTRA_CYCLES + t_wait_cycles + N_cycles_F
    );

    // Now convert to frames per second. The readout time is in f_sys cycles so
    // we can just divide f_sys by t_readout and it will just round down, which
    // is what we want.
    fps = f_sys / t_readout;

    return fps;
}

/*!
 * @brief Compute the maximum frame rate from GUI-visible parameters
 *
 * @param startx    The ROI start column, relative to what is visible to the GUI
 * @param starty    The ROI start row, relative to what is visible to the GUI
 * @param width     The ROI effective width
 * @param height    The ROI effective height
 * @param bit_mode  The bit depth option
 * @param sensopts  Sensor binning and subsampling options
 *
 * @return The maximum number of frames per second
 */
function lince5m_max_fps_setting(
    startx,
    starty,
    width,
    height,
    bit_mode,
    sensopts
) {
    var fps;
    var factor;

    // Get the binning/subsampling factor.
    factor = lince5m_binsubs_factor(sensopts);

    // calculate the ROI needed for the sensor based on user information
    // returns a lince5m_user_to_sensor_result object because Javascript
    // does not have ability to set pointers and need all in object.
    var rtn_ROI = lince5m_user_to_sensor(
        factor,
        startx,
        starty,
        width,
        height,
        bit_mode
    );
    var rtn_DIM = lince5m_get_readout_dims(
        factor,
        rtn_ROI.sensor_x,
        rtn_ROI.sensor_width
    );
    fps = lince5m_get_max_frame_rate(
        rtn_DIM.readout_width / factor,
        rtn_ROI.sensor_height / factor,
        sensopts
    );

    // Return the integer maximum frame rate than can be supported with these parameters.
    fps = fastecMakeInteger(fps);
    return fps;
}

/*!
 * @brief GUI support function to get maximum size for a bit mode
 *
 * @param bit_mode      The bit depth option
 * @param rtn_width     Return maximum permitted width for bit_mode
 * @param rtn_height    Return maximum permitted height for bit_mode
 *
 * @return The maximum number of frames per second
 */
//--------------------------------------------------------------------
// the result object for function lince5m_max_roi_setting()
//--------------------------------------------------------------------
function lince5m_max_roi_setting_result() {
    // object to hold return value for lince5m_max_roi_setting
    this.max_width = 0;
    this.max_height = 0;
}
function lince5m_max_roi_setting(bit_mode) {
    var rtnObj = new lince5m_max_roi_setting_result();

    if (IS_12_BIT_MODE(bit_mode)) {
        rtnObj.max_width = LINCE_MAX_WIDTH_12BIT;
        rtnObj.max_height = LINCE_MAX_HEIGHT_12BIT;
    } else if (IS_10_BIT_MODE(bit_mode)) {
        rtnObj.max_width = LINCE_MAX_WIDTH_10BIT;
        rtnObj.max_height = LINCE_MAX_HEIGHT_10BIT;
    } else {
        /* IS_8_BIT_MODE( bit_mode ) */
        rtnObj.max_width = LINCE_REPORTED_MAX_WIDTH;
        rtnObj.max_height = LINCE_REPORTED_MAX_HEIGHT;
    }

    // return the results
    return rtnObj;
}

/*!
 * @brief Get the binning/subsampling multiplier.
 *
 * @param sensor_opts	Sensor options, which includes binning/subsampling choices
 *
 * @return the binning/subsampling factor
 */
function lince5m_binsubs_factor(sensopts) {
    var factor;

    // GIGE HANDCODE GIGE_SENSOPT_BIN_MASK and GIGE_SENSOPT_SUBS_MASK
    // pull these two bytes out of the lince sensor options
    var bin = fastecHexToDec(csRegPullByte(sensopts, 0));
    var sub = fastecHexToDec(csRegPullByte(sensopts, 1));

    factor = bin * sub; // multiply the values in each part to get a factor

    // Be safe. Be sane.
    if (factor == 0) factor = 1;

    return factor;
}

/*!
 * @brief Get the offset of the capturable area from the max capturable area.
 *
 * @param bit_mode	Bit mode we are capturing with
 * @param sensor_opts	Sensor options, which includes binning/subsampling choices
 * @param *rtn_ofs_x	On return, the "hidden" X offset due to bandwidth control ROI limits
 * @param *rtn_ofs_y	On return, the "hidden" Y offset due to bandwidth control ROI limits
 *
 * @return the binning/subsampling factor
 */
function lince5m_get_ct_offset_result() {
    // object to hold return value for lince5m_get_ct_offset
    this.rtn_ofs_x = 0;
    this.rtn_ofs_y = 0;
}
function lince5m_get_ct_offset(bit_mode, sensor_opts) {
    // make the javascript return object
    var rtnObj = new lince5m_get_ct_offset_result();

    if (IS_12_BIT_MODE(bit_mode) && lince5m_binsubs_factor(sensor_opts) == 1) {
        var rtn_MaxROI = lince5m_max_roi_setting(bit_mode);

        rtnObj.rtn_ofs_x =
            (LINCE_REPORTED_MAX_WIDTH - rtn_MaxROI.max_width) / 2;
        rtnObj.rtn_ofs_y =
            (LINCE_REPORTED_MAX_HEIGHT - rtn_MaxROI.max_height) / 2;
    } else {
        rtnObj.rtn_ofs_x = 0;
        rtnObj.rtn_ofs_y = 0;
    }
    // return the object created in this one to caller so it can
    // have the multiple return values
    return rtnObj;
}

//--------------------------------------------------------------------
// lince_fpn_pixel_support
//
// determines the sensor-specific state of supporting the fpn pixel
//
//--------------------------------------------------------------------
function lince_fpn_pixel_support() {
    return true;
}

//--------------------------------------------------------------------
// lince_max_res_width
//
// determines the sensor-specific maximum width for bitmode
//--------------------------------------------------------------------
function lince_max_res_width(bit_mode) {
    var rtn_MaxROI = lince5m_max_roi_setting(bit_mode);
    return rtn_MaxROI.max_width;
}
//--------------------------------------------------------------------
// lince_max_res_height
//
// determines the sensor-specific maximum height for bitmode
//--------------------------------------------------------------------
function lince_max_res_height(bit_mode) {
    var rtn_MaxROI = lince5m_max_roi_setting(bit_mode);
    return rtn_MaxROI.max_height;
}
