//------------------------------------------------
// fastec_panel.js
//
// Copyright 2018 (c) Fastec Imaging as an unpublished work.
// All Rights Reserved.
//
// The information contained herein is confidential property of
// Fastec Imaging. The use, copying, transfer or disclosure of such
// information is prohibited except by express written agreement
// with Fastec Imaging.
//
// Fastec Imaging Camera Control Web Application
// Panel Objects
//------------------------------------------------

//------------------------------------
// The Fastec Panel Manager Object
//
// This object is responsible for managing
// the state of the panel.
//------------------------------------
function fastecPanelMgr() {
    try {
        // Create our dialogs.
        this.exposureDlg = new fastecExposureDialog();
        this.framerateDlg = new fastecFramerateDialog();
        this.roiDlg = new fastecRoiDialog();

        // Each of the buttons will have a unique object managed
        // together by this container object.
        this.roiTB = new fastecToolbarBtn();
        this.framerateTB = new fastecToolbarBtn();
        this.exposureTB = new fastecToolbarBtn();

        // Used to remember if we need to restore live view.
        this.restoreLiveView = false;

        //---------------------------
        // The common logic for setting up a call.
        //---------------------------
        this.commonEnter = function() {
            // Remember if we need to restore live view.
            if (g_fiwss_liveview.is_streaming()) {
                // Disable live view.
                gFastecAppMgr.onLiveViewToggleTB();
                gPanelMgr.restoreLiveView = true;
            }
        };

        //---------------------------
        // The common logic for finishing up a call.
        //---------------------------
        this.commonExit = function() {
            // Remember if we need to restore live view.
            if (!g_fiwss_liveview.is_streaming() && gPanelMgr.restoreLiveView) {
                // Enable live view.
                gFastecAppMgr.onLiveViewToggleTB();
                gPanelMgr.restoreLiveView = false;
            }
        };

        //---------------------------
        // Done handler for the camera set exposure call.
        //---------------------------
        this.cbhExposureSetDone = function(changedOK) {
            if (!changedOK)
                fastecLogToConsole(
                    gLogAll,
                    "cbhExposureSetDone()" + " changedOK=" + changedOK
                );

            // Handle common cleanup.
            gPanelMgr.commonExit();
        };

        //---------------------------
        // Done handler for the camera set framerate call.
        //---------------------------
        this.cbhFramerateSetDone = function(changedOK) {
            if (!changedOK)
                fastecLogToConsole(
                    gLogAll,
                    "cbhFramerateSetDone()" + " changedOK=" + changedOK
                );

            // Handle common cleanup.
            gPanelMgr.commonExit();
        };

        //---------------------------
        // Done handler for the camera set ROI call.
        //---------------------------
        this.cbhRoiSetDone = function(changedOK) {
            if (!changedOK)
                fastecLogToConsole(
                    gLogAll,
                    "cbhRoiSetDone()" + " changedOK=" + changedOK
                );

            // Handle common cleanup.
            gPanelMgr.commonExit();
        };

        //---------------------------
        // Exposure Button Click Handler
        //---------------------------
        this.onExposureTBClick = function() {
            // Common entry logic.
            gPanelMgr.commonEnter();

            // Make sure the user is OK with this.
            gPanelMgr.exposureDlg.showDialog(gPanelMgr.onExposureDone);
        };

        //---------------------------
        // Framerate Button Click Handler
        //---------------------------
        this.onFramerateTBClick = function() {
            // Common entry logic.
            gPanelMgr.commonEnter();

            // Make sure the user is OK with this.
            gPanelMgr.framerateDlg.showDialog(gPanelMgr.onFramerateDone);
        };

        //---------------------------
        // Done handler for the Expsoure Dialog.
        //---------------------------
        this.onExposureDone = function(ok, new_exposure) {
            fastecLogToConsole(
                gLogTrace,
                "onExposureDone(" + ok + ", " + new_exposure + ")"
            );

            // Was it a cancel?
            if (!ok) {
                // Handle common cleanup.
                gPanelMgr.commonExit();

                // All done.
                return;
            }

            // Write out the new values.
            gCameraData.setCameraExposure(
                new_exposure,
                gPanelMgr.cbhExposureSetDone
            );
        };

        //---------------------------
        // Done handler for the framerate Dialog.
        //---------------------------
        this.onFramerateDone = function(ok, new_framerate) {
            fastecLogToConsole(
                gLogTrace,
                "onFramerateDone(" + ok + ", " + new_framerate + ")"
            );

            // Was it a cancel?
            if (!ok) {
                // Handle common cleanup.
                gPanelMgr.commonExit();

                // All done.
                return;
            }

            // Write out the new values.
            gCameraData.setCameraFramerate(
                new_framerate,
                gPanelMgr.cbhFramerateSetDone
            );
        };

        //---------------------------
        // Done handler for the ROI Dialog.
        //---------------------------
        this.onRoiDone = function(ok, new_width, new_height, new_x, new_y) {
            fastecLogToConsole(
                gLogTrace,
                "onRoiDone(" +
                    ok +
                    ", " +
                    new_width +
                    ", " +
                    new_height +
                    ", " +
                    new_x +
                    ", " +
                    new_y +
                    ")"
            );

            // Was it a cancel?
            if (!ok) {
                // Handle common cleanup.
                gPanelMgr.commonExit();

                // All done.
                return;
            }

            // Write out the new values.
            gCameraData.setCameraROI(
                new_width,
                new_height,
                new_x,
                new_y,
                gPanelMgr.cbhRoiSetDone
            );
        };

        //---------------------------
        // ROI Button Click Handler
        //---------------------------
        this.onRoiTBClick = function() {
            // Common entry logic.
            gPanelMgr.commonEnter();

            // Make sure the user is OK with this.
            gPanelMgr.roiDlg.showDialog(gPanelMgr.onRoiDone);
        };

        //---------------------------
        // Panel initialization
        //---------------------------
        this.initPanel = function() {
            this.roiTB.setTBSelector($("#editROI"));
            this.framerateTB.setTBSelector($("#editFramerate"));
            this.exposureTB.setTBSelector($("#editExposure"));

            // Initialize states of buttons to be active
            this.roiTB.setActiveState(true);
            this.roiTB.setVisibleState(true);
            this.roiTB.setClickHandler(gPanelMgr.onRoiTBClick);

            this.framerateTB.setActiveState(true);
            this.framerateTB.setVisibleState(true);
            this.framerateTB.setClickHandler(gPanelMgr.onFramerateTBClick);

            this.exposureTB.setActiveState(true);
            this.exposureTB.setVisibleState(true);
            this.exposureTB.setClickHandler(gPanelMgr.onExposureTBClick);
        };

	//---------------------------------
	// Update the UI elements of the panel.
	//---------------------------------
	this.updateUI = function() {
	    switch (gCameraData.sysCameraState)
	    {
		case gRegs.GIGE_CAMERA_STATE_REVIEW:
		    // Disable the settings buttons.
		    $("#editROI").addClass("ui-state-disabled");
		    $("#editFramerate").addClass("ui-state-disabled");
		    $("#editExposure").addClass("ui-state-disabled");
		    break;
		case gRegs.GIGE_CAMERA_STATE_ARMED:
		case gRegs.GIGE_CAMERA_STATE_DEARMING:
		case gRegs.GIGE_CAMERA_STATE_TRIGGERED:
		    // Allow exposure, but not the others.
		    $("#editExposure").removeClass("ui-state-disabled");
		    $("#editROI").addClass("ui-state-disabled");
		    $("#editFramerate").addClass("ui-state-disabled");
		    break;
		default:
		    // Allow settings to be modified.
		    $("#editROI").removeClass("ui-state-disabled");
		    $("#editFramerate").removeClass("ui-state-disabled");
		    $("#editExposure").removeClass("ui-state-disabled");
		    break;
	    }
	};
    } catch (err) {
        var theDetails = "caught: fastecPanelMgr-" + err.description;
        gFaultHandler.logError(theDetails);
    }
}

//------------------------------------
// The Fastec ROI Dialog
//------------------------------------
function fastecRoiDialog() {
    try {
        // Who to callback when dialog closes -
        // callback signature takes in OK/Cancel button boolean, width, height, x-offset and y-offset.
        this.callbackMethod = null;

        // Used to determine if bindings have taken place.
        this.initializedBindings = false;

        // Used to track the state of the auto-center checkbox.
        this.auto_center = true;

        //--------------------------------------
        // Auto-center button click handler
        //--------------------------------------
        this.onCenterClick = function() {
            gPanelMgr.roiDlg.auto_center = !gPanelMgr.roiDlg.auto_center;
            if (gPanelMgr.roiDlg.auto_center) {
                // We're centering, so recalculate width/height points.
                gPanelMgr.roiDlg.onHeightStop(null, null);
                gPanelMgr.roiDlg.onWidthStop(null, null);
            } else {
                var x_max = $("#dlgROI_X_offset").prop("max");
                var y_max = $("#dlgROI_Y_offset").prop("max");

                // Only enable if our max is greater than 0.
                if (x_max > 0) $("#dlgROI_X_offset").slider("enable");
                if (y_max > 0) $("#dlgROI_Y_offset").slider("enable");
            }
        };

        //--------------------------------------
        // OK button click handler
        //--------------------------------------
        this.onOkClick = function() {
            if (gPanelMgr.roiDlg.callbackMethod != null) {
                var new_width = $("#dlgROI_width").val();
                var new_height = $("#dlgROI_height").val();
                var new_x = $("#dlgROI_X_offset").val();
                var new_y = $("#dlgROI_Y_offset").val();

                gPanelMgr.roiDlg.callbackMethod(
                    true,
                    new_width,
                    new_height,
                    new_x,
                    new_y
                );
                gPanelMgr.roiDlg.callbackMethod = null;
            }
        };

        //--------------------------------------
        // Cancel button click handler
        //--------------------------------------
        this.onCancelClick = function() {
            if (gPanelMgr.roiDlg.callbackMethod != null) {
                gPanelMgr.roiDlg.callbackMethod(false);
                gPanelMgr.roiDlg.callbackMethod = null;
            }
        };

        //--------------------------------------
        // Height slider handler
        //--------------------------------------
        this.onHeightStop = function(e, ui) {
            var new_height = $("#dlgROI_height").val();
            var new_max = gCameraData.maxResHeight - new_height;

            // A maximum of zero is a special case...
            if (new_max == 0) {
                // If the new maximum is zero, then we want to
                // make sure the value is set to 0, update the maximum,
                // and disable the control.
                $("#dlgROI_Y_offset")
                    .val(0)
                    .slider("refresh");
                $("#dlgROI_Y_offset")
                    .prop("max", new_max)
                    .slider("refresh");
                $("#dlgROI_Y_offset").slider("disable");
            } else {
                // Update the maximum value for the slider.
                $("#dlgROI_Y_offset")
                    .prop("max", new_max)
                    .slider("refresh");

                if (gPanelMgr.roiDlg.auto_center) {
                    var y_offset =
                        gCameraData.maxResHeight / 2 - new_height / 2;

                    fastecLogToConsole(gLogAll, "y_offset:" + y_offset);
                    if (y_offset & 1) y_offset -= 1;

                    $("#dlgROI_Y_offset")
                        .val(y_offset)
                        .slider("refresh");
                    $("#dlgROI_Y_offset").slider("disable");
                } else {
                    // User can modify, so enable the slider.
                    $("#dlgROI_Y_offset").slider("enable");
                }
            }
        };

        //--------------------------------------
        // Width slider handler
        //--------------------------------------
        this.onWidthStop = function(e, ui) {
            var new_width = $("#dlgROI_width").val();
            var new_max = gCameraData.maxResWidth - new_width;

            // A maximum of zero is a special case...
            if (new_max == 0) {
                // If the new maximum is zero, then we want to
                // make sure the value is set to 0, update the maximum,
                // and disable the control.
                $("#dlgROI_X_offset")
                    .val(0)
                    .slider("refresh");
                $("#dlgROI_X_offset")
                    .prop("max", new_max)
                    .slider("refresh");
                $("#dlgROI_X_offset").slider("disable");
            } else {
                // Update the maximum value for the slider.
                $("#dlgROI_X_offset")
                    .prop("max", new_max)
                    .slider("refresh");

                if (gPanelMgr.roiDlg.auto_center) {
                    var x_offset = gCameraData.maxResWidth / 2 - new_width / 2;

                    fastecLogToConsole(gLogAll, "x_offset:" + x_offset);
                    if (x_offset & 1) x_offset -= 1;

                    $("#dlgROI_X_offset")
                        .val(x_offset)
                        .slider("refresh");
                    $("#dlgROI_X_offset").slider("disable");
                } else {
                    // User can modify, so enable the slider.
                    $("#dlgROI_X_offset").slider("enable");
                }
            }
        };

        //-----------------------------------
        //-----------------------------------
        this.initBindings = function() {
            // Give our object a way to bind events to it
            this.myObjData = { fiRoiDialogOBJ: this };

            // Bind handlers for the OK and Cancel buttons.
            $("#dlgROI_OK").bind("click", this.myObjData, this.onOkClick);
            $("#dlgROI_Cancel").bind(
                "click",
                this.myObjData,
                this.onCancelClick
            );
            $("#div-width-slider").bind(
                "change",
                this.myObjData,
                this.onWidthStop
            );
            $("#div-height-slider").bind(
                "change",
                this.myObjData,
                this.onHeightStop
            );
            $("#dlgROI_center").bind(
                "click",
                this.myObjData,
                this.onCenterClick
            );

            this.initializedBindings = true; // we are initialized
        };

        //---------------------------
        // Shows modal dialog
        //---------------------------
        this.showDialog = function(inCallbackFunc) {
            // Save the callback information.
            this.callbackMethod = inCallbackFunc;

            // If haven't initialized bindings do so.
            if (!this.initializedBindings) this.initBindings();

            // Always start out with the "auto-center" checkbox checked.
            $("#dlgROI_center")
                .prop("checked", this.auto_center)
                .checkboxradio("refresh");

            // Initialize our current settings.
            $("#dlgROI_width")
                .prop("max", gCameraData.maxResWidth)
                .slider("refresh");
            $("#dlgROI_width")
                .prop("min", gCameraData.minROIWidth)
                .slider("refresh");
            $("#dlgROI_width")
                .val(gCameraData.recROIWidth)
                .slider("refresh");

            $("#dlgROI_height")
                .prop("max", gCameraData.maxResHeight)
                .slider("refresh");
            $("#dlgROI_height")
                .prop("min", gCameraData.minROIHeight)
                .slider("refresh");
            $("#dlgROI_height")
                .val(gCameraData.recROIHeight)
                .slider("refresh");

            // For the offsets, we always start out disabled, because we always
            // start with auto-center enabled.
            //
            //========================================================================
            // TODO: We really need to check to see if we're centered, rather than
            // 	     simply assuming...
            //========================================================================
            var calc_max = gCameraData.maxResWidth - gCameraData.recROIWidth;
            $("#dlgROI_X_offset")
                .val(gCameraData.recOffsetX)
                .slider("refresh");
            $("#dlgROI_X_offset")
                .prop("max", calc_max)
                .slider("refresh");
            $("#dlgROI_X_offset").slider("disable");

            calc_max = gCameraData.maxResHeight - gCameraData.recROIHeight;
            $("#dlgROI_Y_offset")
                .val(gCameraData.recOffsetY)
                .slider("refresh");
            $("#dlgROI_Y_offset")
                .prop("max", calc_max)
                .slider("refresh");
            $("#dlgROI_Y_offset").slider("disable");

            // show the message box
            $("#dlgROI").popup("open");
        };
    } catch (err) {
        var theDetails = "caught: dlgROI-" + err.description;
        gFaultHandler.logError(theDetails);
    }
}

//------------------------------------
// The Fastec Framerate Dialog
//------------------------------------
function fastecFramerateDialog() {
    try {
        // Who to callback when dialog closes -
        // callback signature takes in OK/Cancel button boolean, framerate value.
        this.callbackMethod = null;

        // Used to determine if bindings have taken place.
        this.initializedBindings = false;

        //--------------------------------------
        // Framerate slider handler
        //--------------------------------------
        this.onChange = function(e, ui) {
            var new_framerate = $("#dlgFramerate_val").val();
            var fr_msec = fastecMakeInteger(MILLI_PER / new_framerate);
            $("#dlgFramerate_val-label").text(
                "Framerate (FPS) [" + fr_msec + " mSec]:"
            );
        };

        //--------------------------------------
        // OK button click handler
        //--------------------------------------
        this.onOkClick = function() {
            if (gPanelMgr.framerateDlg.callbackMethod != null) {
                var new_framerate = $("#dlgFramerate_val").val();

                gPanelMgr.framerateDlg.callbackMethod(true, new_framerate);
                gPanelMgr.framerateDlg.callbackMethod = null;
            }
        };

        //--------------------------------------
        // Cancel button click handler
        //--------------------------------------
        this.onCancelClick = function() {
            if (gPanelMgr.framerateDlg.callbackMethod != null) {
                gPanelMgr.framerateDlg.callbackMethod(false, null);
                gPanelMgr.framerateDlg.callbackMethod = null;
            }
        };

        //-----------------------------------
        //-----------------------------------
        this.initBindings = function() {
            // Give our object a way to bind events to it
            this.myObjData = { fiFramerateDialogOBJ: this };

            // Bind handlers for the OK and Cancel buttons.
            $("#dlgFramerate_OK").bind("click", this.myObjData, this.onOkClick);
            $("#dlgFramerate_Cancel").bind(
                "click",
                this.myObjData,
                this.onCancelClick
            );
            $("#div-framerate-slider").bind(
                "change",
                this.myObjData,
                this.onChange
            );

            this.initializedBindings = true; // we are initialized
        };

        //---------------------------
        // Shows modal dialog
        //---------------------------
        this.showDialog = function(inCallbackFunc) {
            // Save the callback information.
            this.callbackMethod = inCallbackFunc;

            // If haven't initialized bindings do so.
            if (!this.initializedBindings) this.initBindings();

            // Initialize our current settings.
            var max_rate = gCameraData.max_framerate();
            var fr_msec = fastecMakeInteger(MILLI_PER / gCameraData.recFPS);
            fastecLogToConsole(gLogAll, "max_frame_rate = " + max_rate);
            $("#dlgFramerate_val")
                .prop("max", max_rate)
                .slider("refresh");
            $("#dlgFramerate_val")
                .val(gCameraData.recFPS)
                .slider("refresh");
            $("#dlgFramerate_val-label").text(
                "Framerate (FPS) [" + fr_msec + " mSec]:"
            );

            // show the message box
            $("#dlgFramerate").popup("open");
        };
    } catch (err) {
        var theDetails = "caught: dlgROI-" + err.description;
        gFaultHandler.logError(theDetails);
    }
}

//------------------------------------
// The Fastec Exposure Dialog
//------------------------------------
function fastecExposureDialog() {
    try {
        // Who to callback when dialog closes -
        // callback signature takes in OK/Cancel button boolean, exposure value.
        this.callbackMethod = null;

        // Used to determine if bindings have taken place.
        this.initializedBindings = false;

        //--------------------------------------
        // OK button click handler
        //--------------------------------------
        this.onOkClick = function() {
            if (gPanelMgr.exposureDlg.callbackMethod != null) {
                var new_exposure = $("#dlgExposure_val").val();

                gPanelMgr.exposureDlg.callbackMethod(true, new_exposure);
                gPanelMgr.exposureDlg.callbackMethod = null;
            }
        };

        //--------------------------------------
        // Cancel button click handler
        //--------------------------------------
        this.onCancelClick = function() {
            if (gPanelMgr.exposureDlg.callbackMethod != null) {
                gPanelMgr.exposureDlg.callbackMethod(false, null);
                gPanelMgr.exposureDlg.callbackMethod = null;
            }
        };

        //-----------------------------------
        //-----------------------------------
        this.initBindings = function() {
            // Give our object a way to bind events to it
            this.myObjData = { fiExposureDialogOBJ: this };

            // Bind handlers for the OK and Cancel buttons.
            $("#dlgExposure_OK").bind("click", this.myObjData, this.onOkClick);
            $("#dlgExposure_Cancel").bind(
                "click",
                this.myObjData,
                this.onCancelClick
            );

            this.initializedBindings = true; // we are initialized
        };

        //---------------------------
        // Shows modal dialog
        //---------------------------
        this.showDialog = function(inCallbackFunc) {
            // Save the callback information.
            this.callbackMethod = inCallbackFunc;

            // If haven't initialized bindings do so.
            if (!this.initializedBindings) this.initBindings();

            // Initialize our current settings.
            fastecLogToConsole(
                gLogAll,
                "exposure = " +
                    gCameraData.sys_min_shutter +
                    "-" +
                    gCameraData.sys_max_shutter
            );
            $("#dlgExposure_val")
                .prop("min", gCameraData.sys_min_shutter)
                .slider("refresh");
            $("#dlgExposure_val")
                .prop("max", gCameraData.sys_max_shutter)
                .slider("refresh");
            $("#dlgExposure_val")
                .val(gCameraData.recShutter)
                .slider("refresh");

            // show the message box
            $("#dlgExposure").popup("open");
        };
    } catch (err) {
        var theDetails = "caught: dlgROI-" + err.description;
        gFaultHandler.logError(theDetails);
    }
}
