/*!
 * @file fastec_math.js
 * @brief This file contains the implementation of the camera 
 *        math core routines. It should directly correspond to
 *        math classes used in the camera server.
 *
 * Copyright 2018 (c) Fastec Imaging as an unpublished work.
 * All Rights Reserved.
 *
 * The information contained herein is confidential property of
 * Fastec Imaging. The use, copying, transfer or disclosure of such
 * information is prohibited except by express written agreement
 * with Fastec Imaging.
 */

//--------------------------------------------------------------------
// Constant, Macro and Type Definitions
//--------------------------------------------------------------------

//! The number of pixels per kernel.
var SENSOR_KERN_WIDTH = 24;

//! Sensor clock 63 Mhz.
var CLOCK = 63000000;

//! Frame overhead time, in CLOCKs.
var FOT = 465.0;

//! Row overhead time.
var ROT = 12.0;

//! The number of bytes in an SSD sector.
var SSD_SECTOR = 512;

//! The number of bytes in an SSD page.
var SSD_PAGE = 8 * SSD_SECTOR;

//! The number of bytes in a QuadWord.
var QUADWORD_BYTES = 16;

//! The per-frame overhead required for frame read-out, in microseconds.
var DEFAULT_FRAME_OVERHEAD = 6;

//! How many QWORDS are in a gigabyte.
var GBYTES_TO_QWORDS = 64 * 1024 * 1024;

//! Define our increment in terms of QUADWORDs. The value is
//! calculated as 256 MBytes in quadwords because we are representing
//! a session length "quantum" as 1/4 of 1 GByte = 256 MBytes.
var SESSION_INCR_QWORDS = 256 / 16 * 1024 * 1024;

//! The conversion factor from sectors to QuadWords.
var SSD_SECTOR_LEN_QWORDS = 32; // (SSD_SECTOR / QUADWORD_BYTES)

//! Conversion to use when going from seconds to microseconds.
var USEC_CONVERSION = 1000000;

var MILLI_PER = 1000; //!< How many milli-somethings make up a something.
var MICRO_PER = MILLI_PER * MILLI_PER; //!< How many micro things make up a thing.

//--------------------------------------------------------------------
// calculate quadwords in use
//--------------------------------------------------------------------
function bytes_to_quadwords(in_bytesToConvert) {
    var quadWords = in_bytesToConvert * SSD_SECTOR_LEN_QWORDS;
    return quadWords;
}

/*!
 * @brief Calculate the maximum shutter speed based upon a frame rate.
 *
 * @param framerate         Frame rate to use to calculate the maximum shutter speed.
 * @param frame_overhead    Per-frame overhead.
 *
 * @return 0 if the roi not valid, or the max frame rate if it is.
 */
function calc_max_exposure(framerate, frame_overhead) {
    // A frame overhead of 0 is not allowed.
    if (!frame_overhead) frame_overhead = DEFAULT_FRAME_OVERHEAD;

    // A frame rate of 0 is not allowed.
    if (framerate == 0) return 0;

    var max_exposure = USEC_CONVERSION / framerate - frame_overhead;

    // make the max_exposure an interger don't want fractional data
    max_exposure = fastecMakeInteger(max_exposure);

    // Return the maximum exposure for the given framerate.
    return max_exposure;
}

/*!
 * @brief Calculate how many frames will fit in the partition given a given width and height.
 *
 * @param part_size_quads   The partition size in quadwords or sectors.
 * @param width             The width we want to use for the math.
 * @param height            The height we want to use for the math.
 * @param bit_mode          Which bit mode are we using.
 * @param ltr               Are we in ltr mode or not.
 *
 * @return The number of frames which will fit in the partition.
 */
function calc_frame_count(part_size_quads, width, height, bit_mode, ltr) {
    var frame_count;
    var value;

    if (ltr) {
        var frame_space;

        // If we are in LTR mode then the calculation is different.

        // Get size in bytes.
        frame_space = calc_ltr_frame_size(width, height, bit_mode);
        if (frame_space == 0) {
            return 1;
        }

        // Do unit conversion to get number of frames that fit in the available space.
        frame_count = fastecMakeInteger(part_size_quads * QUADWORD_BYTES);
        frame_count = fastecMakeInteger(frame_count / frame_space);
    } else {
        var frame_space;

        // Initialize our variable.
        value = width;

        // Calculate the number of bytes in a row.
	if (IS_12_BIT_MODE(bit_mode)) {
	    // For 12-bit pixels, we have 5 pixels per 64-bit double
	    // word with 4 bits left over.
	    if ((value % 5) != 0) {
		value = (value / 5) + 1;
	    } else {
		value = value / 5;
	    }
	    value = fastecMakeInteger(value);
	    value = value * 8; // now make it bytes
	} else if (IS_10_BIT_MODE(bit_mode)) {
            // In 10-bit mode, 3 pixels fit in a 32 bit word with 2 bits left over.
            if ((value % 3) != 0) {
		value = value / 3 + 1;
	    } else {
		value = value / 3;
	    }
            value = fastecMakeInteger(value); // make sure integer not floating point
            value = value * 4; // now make it bytes
        } else {
            // If the bit width is 8 bits, then 4 pixels fit in a 32 bit word
            if ((value % 4) == 0) {
		value = value / 4;
	    } else {
		value = value / 4 + 1;
	    }
            value = fastecMakeInteger(value); // javascript necessity
            value = value * 4; // put it back in bytes
        }

        // Calculate the number of bytes per frame.
        frame_space = value * height;

	// Round the frame size up to hexword size.
        frame_space = (frame_space + 63) / 64;
        frame_space = fastecMakeInteger(frame_space);
	frame_space = frame_space * 64;

	// Account for the metadata.
	frame_space = frame_space + 64;

        // Now we have partition space and the space for a frame,
        // calculate the number of frames that can be created.
        frame_count = (part_size_quads * QUADWORD_BYTES) / frame_space;

        // make the frames into an integer
        frame_count = fastecMakeInteger(frame_count);
    }

    return frame_count;
}

/*!
 * @brief This routine calculates the frame size on [page boundaries](@ref CLIENT_MATH::SSD_PAGE)
 *        given an ROI and the bit depth.
 *
 * @param width         The width of the image.
 * @param height        The height of the image.
 * @param bit_depth     The bit-depth used (GIGE_BITMODE_10_BITS, GIGE_BITMODE_LOWER_8_BITS, etc.)
 *
 * @return The frame size, in [pages](@ref CLIENT_MATH::SSD_PAGE).
 */
// this routine calculates the frame size on sector boundaries given an roi and the bit depth
function calc_ltr_frame_size(width, height, bit_depth) {
    var frame_size = width;
    var frame_size_temp = 0;

    // Sanity check.
    if (width == 0 || height == 0) {
        return 0;
    }

    // Get the bit-depth we're capture at.
    if (IS_12_BIT_MODE(bit_depth)) {
	// For 12-bit pixels, we have 5 pixels per 64-bit double
	// word with 4 bits left over.
	frame_size_temp = frame_size % 5;
	if (frame_size_temp != 0) {
	    frame_size = (frame_size / 5) + 1;
	} else {
	    frame_size = frame_size / 5;
	}
        frame_size = fastecMakeInteger(frame_size);
        frame_size = frame_size * 8; // now make it bytes
    }
    else if (IS_10_BIT_MODE(bit_depth)) {
        // For 10-bit pixels, we have 3 pixels per 32-bit word.
        // We use an even number of 32-bit words per row.
        frame_size_temp = frame_size % 3;
        if (frame_size_temp != 0) {
	    frame_size = (frame_size / 3) + 1;
	} else {
	    frame_size = frame_size / 3;
	}
        frame_size = fastecMakeInteger(frame_size);
        frame_size = frame_size * 4; // now make it bytes
    } else {
        // For 8-bit pixels, we have 4 pixels per 32-bit word.
        // Make sure there is an even number of 32-bit words per row.
        frame_size_temp = frame_size % 4;
        if (frame_size_temp == 0) {
	    frame_size = frame_size / 4;
	} else {
	    frame_size = (frame_size / 4) + 1;
	}
        frame_size = fastecMakeInteger(frame_size);
        frame_size = frame_size * 4; // put it back in bytes
    }

    // Multiple the row size by the number of rows.
    frame_size = frame_size * height;

    // Round the frame size up to a hexword size.
    frame_size = fastecMakeInteger((frame_size + 63) / 64);
    frame_size = frame_size * 64;

    // Add space for the per-frame metadata.
    frame_size = frame_size + 64;

    // Return our frame size.
    return frame_size;
}

/*!
 * @brief Given a number of [sectors](@ref CLIENT_MATH::SSD_SECTOR], return
 *        the number of [QWORD](@ref CLIENT_MATH::SSD_SECTOR_LEN_QWORDS).
 *
 * @param sector_count  The number of sectors.
 * @return The number of QWords.
 */
function calc_ltr_part_size(sector_count) {
    // keep the math functions in client_math.cpp of fasMotion and this
    // functionality is exactly that of the bytes_to_quadwords above
    return bytes_to_quadwords(sector_count);
}

/*!
 * @brief Calculate the maximum frame rate for a given width and height.
 *        This is a general interface and sensor specific implementations
 *        are handled at a sub layer in fastec_sensor file.
 *
 * @param startx                The start of the ROI in pixels.
 * @param width                 The width of the ROI in pixels.
 * @param height                The height of the ROI.
 * @param cam_max_framerate     The maximum licensed frame rate of the camera.
 * @param bit_depth             The number of bits per pixel.
 * @param sensopts              The sensor options (currently gCameraData.recMenuBinSample)
 * @param ltr                   True if ltr mode is enabled.
 * @param ssd_bandwidth         The maximum number of bytes per second allowed to the SSD. (gCameraData.theSSDBandWidth)
 *
 * @return The maximum allowable frame rate for the given ROI.
 */
function calc_max_framerate(
    startx,
    width,
    height,
    bit_depth,
    sensopts,
    ltr,
    ssd_bandwidth
) {
    var max_framerate;

    // Initialize our rect.
    var roi_x = startx;
    var roi_y = 0;
    var roi_width = width;
    var roi_height = height;

    // Get the frame rate.
    max_framerate = sensor_max_frame_rate(
        roi_x,
        roi_y,
        roi_width,
        roi_height,
        bit_depth,
        sensopts
    );

    // Are we in LTR mode?
    if (ltr) {
        var frame_size;
        var max_ltr_framerate;

        // if we are in ltr mode then the calculation is different
        frame_size = calc_ltr_frame_size(roi_width, roi_height, bit_depth);
        if (frame_size == 0) {
            return 1;
        }

        max_ltr_framerate = fastecMakeInteger(ssd_bandwidth / frame_size);

        // Make sure our LTR maximum is the lower of the two.
        if (max_ltr_framerate < max_framerate)
            max_framerate = max_ltr_framerate;
    }

    // Return our framerate (FPS = Frames Per Second)
    return max_framerate;
}

//------------------------------------
// UTILITY:  convert an integer to QWord (32 bytes)
// input:
//      byteData - the data to convert in bytes
// output:
//      quadData - the data converted to quad words
//------------------------------------
function convert_to_QWORD(byteData) {
    var QUADWORD_SIZE = 16;
    var myModVal;
    var qwordData;

    myModVal = byteData % QUADWORD_SIZE;
    qwordData = byteData / QUADWORD_SIZE;
    qwordData = fastecMakeInteger(qwordData);

    if (myModVal != 0) qwordData++;

    // make sure it's a multiple of 32 bytes
    myModVal = qwordData & 2;
    if (myModVal != 0) qwordData++;
    return qwordData;
}

//------------------------------------
// UTILITY:  Calculates the ratio of a given framerate and shutter speed
// input:
//      exp - the shutter speed in microseconds
//      fr - the frame rate in frames per second
// output:
//      nx - The calculated ratio. If an error occurs return < 0
//------------------------------------
function recexp_calc_nx(exp, fr) {
    var nx;

    // Sanity check.
    if (exp == 0 || fr == 0) {
        return -1;
    }

    // Convert to an integer and return.
    nx = MICRO_PER * 10 / (exp * fr);
    nx = (nx + 9) / 10;

    // because this is javascript must call handler to convert to integer
    nx = fastecMakeInteger(nx);

    return nx;
}

//------------------------------------
// UTILITY:  Calculates the ratio of a given framerate and shutter speed
// input:
//      nx - The ratio of the shutter speed to the framerate (1X, 2X, 4x . . .)
//      fr - the frame rate in frames per second
// output:
//      exp_us - The calculated exposure in microseconds
//------------------------------------
function recexp_calc_exp(nx, fr) {
    var exp_us; // The integration time in micro seconds.

    // Sanity check
    if (nx <= 0 || fr == 0) {
        return 0;
    }

    // Calculate the adjusted frame time in microseconds.
    exp_us = MICRO_PER / fr;

    // If we are trying for the longest exposure time (1X), then
    // we need to back the duration off by the frame overhead.
    if (nx == 1) exp_us = exp_us - gCameraData.sysFrmOverhead;
    else {
        // Calculate the integration time in microseconds.
        exp_us = exp_us / nx;
    }

    // Sanity check the result.
    if (exp_us < gCameraData.sys_min_shutter)
        exp_us = gCameraData.sys_min_shutter;

    // translate into integer because of javascript
    exp_us = fastecMakeInteger(exp_us);

    // Return the exposure duration.
    return exp_us;
}

//------------------------------------
// UTILITY:  Calculates the microseconds used per frame
// input:
//      frame_rate - the frame rate in frames per second
// output:
//      msPerFrame - The calculated microseconds per frame
//------------------------------------
function calc_ms_per_frame(frame_rate) {
    var msPerFrame = MICRO_PER / frame_rate;
    return msPerFrame;
}
