//------------------------------------------------
// fastec_fcp_io.js
//
// The Fastec Imaging Series3 Web Application
// FCP Input Output Support Object -
//     Provides Interface between Camera Object and the
//     WSS FCP Object
//
// Copyright 2010-2018 (c) Fastec Imaging as an unpublished work.
// All Rights Reserved.
//
// The information contained herein is confidential property of
// Fastec Imaging. The use, copying, transfer or disclosure of such
// information is prohibited except by express written agreement
// with Fastec Imaging.
//
// Fastec Imaging Camera Control Web Application
// Web Socket Server Interface Functions for Application
//------------------------------------------------
var gInternalDataPkgID = 0;

//------------------------------------
// The Fastec FCP Input/Ouput Module
// Responsible for managing all WSS
// communications for the FCP data
// exchange packets across a connection
//------------------------------------
function fiFCP_IO(
    in_isListener, // 0 if load/store or 1 if listener
    in_cbhConnected, // null or cbh if user wants to know when connected
    in_cbhDisconnected // null or cbh if user wants to know when disconnected
) {
    fifcp_str = {
        wss_fcpInfoTitle: "FCP State: ",
        wss_listenerTitle: "Listener State: ",

        wss_fcpInfoError: "Request Completed in Error",
        wss_fcpInfoSuccess: "Request Completed Successfully",
        wss_fcpInfoNew: "Uninitialized FCP Connection",
        wss_fcpInfoSetupErr: "FCP Request Was Invalid",
        wss_fcpInfoPending: "Waiting on Previous FCP Request",
        wss_fcpInfoReady: "Ready and Waiting for Request",
        wss_fcpInfoNoWSSConn: "No FCP WSS Connection Available",
        wss_fcpInfoWSSBusy: "FCP WSS Busy"
    };

    try {
        // to allow for access to the instance of this object during a cb
        // callback handler - because the "this" object will not be available
        var me = this;

        // internal states of this FCP IO object
        var fiIO_States = {
            IONew: 0, // this object is in new uninitialized state
            IOReady: 1, // the object is ready for requests
            IOSetupErr: 2, // input data for IO request was invalid
            IOPending: 4, // there is a pending request sent on IO channel?
            IOError: 8, // this object received an error in response to request
            IOSuccess: 16, // this object received a successful response to request
            IONoWSSConn: 32, // this object has no existing WSS connection
            IOWSSBusy: 64 // the WSS is reporting it is busy with another request
        };
        this.ioState = fiIO_States.IONew; // default to uninitialized state

        // save the cbh for wss connection passed from caller or set null
        this.cbhAfterWSSConnects = null;
        if (in_cbhConnected != null) this.cbhAfterWSSConnects = in_cbhConnected;

        // save the cbh for wss disconnection passed from caller or set null
        this.cbhWSSDisconnected = null;
        if (in_cbhDisconnected != null)
            this.cbhWSSDisconnected = in_cbhDisconnected;

        //--------------------------------------
        // when the managed wss object is connected to
        // the camera it calls back here
        //--------------------------------------
        this.cbhWSSConnected = function() {
            // if this object has a cbh from user invoke it
            if (me.cbhAfterWSSConnects != null) me.cbhAfterWSSConnects(); // make the callback
        };
        //--------------------------------------
        // when the managed wss object is disconnected from
        // the camera it calls back here
        //--------------------------------------
        this.cbhWSSDisconnected = function() {
            // if this object has a cbh from user invoke it
            if (me.cbhAfterWSSDisconnects != null) me.cbhAfterWSSDisconnects(); // make the callback
        };

        // is this object a listener object? 0 means no 1 means yes
        this.isListener = in_isListener;

        // Each FCP IO object has its own connection to the Fastec
        // Web Socket Server object and is responsible only for
        // communications on that WSS interface
        this.fcpWSS = new fiwss_fcp(
            this.isListener,
            this.cbhWSSConnected,
            this.cbhWSSDisconnected
        );

        // Create the socket.
        this.fcpWSS.create_wss_socket();

        // each FCP IO object has its own data package object
        // that contains the request and response information
        // from the FCP WSS connection
        this.fcpDataPkg = null;

        // manages a callback handler (cbh) for the field update
        // if this is a load request -- the interface for this cbh:
        //
        // result = cbhFieldUpdate(async_msg, field_address, field_data_loaded)
        // the update field handler should return true if updated or
        // false if did not update the field data.
        this.cbhFieldUpdate = null;

        // manages a callback handler (cbh) to invoke when a response
        // data package is completed for a request -- interface:
        //
        // result = cbhRequestCompleted(request_successful_boolean)
        // the callback handler will notify the requester whether or
        // not the request was successful (true or false) and where called from
        this.cbhRequestCompleted = null;

        // should be able to provide more information on any
        // request operation using the response data in the
        // data package.  This allows more details reported to
        // user whether error or success conditions.
        // it returns a user-displayable message for the generic
        // messaging system
        this.getTheStatusMsg = function() {
            var userMsg;

            // pop up the right title depending on how this is used
            if (this.isListener == 1) userMsg = fifcp_str.wss_listenerTitle;
            else userMsg = fifcp_str.wss_fcpInfoTitle;

            if (this.ioState == fiIO_States.IONew)
                userMsg = userMsg + fifcp_str.wss_fcpInfoNew;
            else if (this.ioState == fiIO_States.IOSetupErr)
                userMsg = userMsg + fifcp_str.wss_fcpInfoSetupErr;
            else if (this.ioState == fiIO_States.IOPending)
                userMsg = userMsg + fifcp_str.wss_fcpInfoPending;
            else if (this.ioState == fiIO_States.IOError)
                userMsg = userMsg + fifcp_str.wss_fcpInfoError;
            else if (this.ioState == fiIO_States.IOSuccess)
                userMsg = userMsg + fifcp_str.wss_fcpInfoSuccess;
            else if (this.ioState == fiIO_States.IOReady)
                userMsg = userMsg + fifcp_str.wss_fcpInfoReady;
            else if (this.ioState == fiIO_States.IONoWSSConn)
                userMsg = userMsg + fifcp_str.wss_fcpInfoNoWSSConn;
            else if (this.ioState == fiIO_States.IOWSSBusy)
                userMsg = userMsg + fifcp_str.wss_fcpInfoWSSBusy;

            // pull out the response string if one is available.
            if (this.fcpDataPkg != null) {
                var tempBuffer = "";

                // add the status information string
                if (this.fcpDataPkg.jsonResponseData != null) {
                    if (
                        this.fcpDataPkg.jsonResponseData.pkginfo.status_info !=
                        undefined
                    ) {
                        tempBuffer = this.fcpDataPkg.jsonResponseData.pkginfo
                            .status_info;
                        userMsg = userMsg + "<br>" + tempBuffer;

                        if (this.fcpDataPkg.jsonRequestData != undefined) {
                            tempBuffer = this.fcpDataPkg.jsonRequestData;
                            userMsg = userMsg + "<br><br>" + tempBuffer;
                        }
                    }
                }
            }

            return userMsg; // send back the string.
        };

        //--------------------------------------
        // Provide a close WSS connection
        //--------------------------------------
        this.closeWSS = function() {
            if (this.fcpWSS !== null) this.fcpWSS.closeWSS();
        };

        //--------------------------------------
        // reset this object for new Camera I/O
        //--------------------------------------
        this.resetIO = function(in_forceIt) {
            var canReset = true;
            if (this.ioState == fiIO_States.IOPending) {
                if (!in_forceIt) canReset = false;
            }
            // if the object can be reset then do so
            if (canReset) {
                // if the objects have been initialized reset to Ready State
                if (this.fcpWSS != null && this.fcpDataPkg != null)
                    this.ioState = fiIO_States.IOReady;
                else this.ioState = fiIO_States.IONew;

                // reset the callback handler interface
                this.cbhFieldUpdate = null;
                this.cbhRequestCompleted = null;
            }
        };

        //--------------------------------------
        // Does this object have a WSS connection?
        //--------------------------------------
        this.hasWSSConnection = function() {
            var hasWSSConnection = false;
            if (this.fcpWSS != null) {
                if (this.fcpWSS.is_connected()) hasWSSConnection = true;
            }

            return hasWSSConnection;
        };

        //--------------------------------------
        // Is there a Request Pending
        //--------------------------------------
        this.isIOPending = function() {
            if (this.ioState == fiIO_States.IOPending) return true;
            else return false;
        };

        //--------------------------------------
        // Was the Request Successful?
        //--------------------------------------
        this.isIOSuccess = function() {
            if (this.ioState == fiIO_States.IOSuccess) return true;
            else return false;
        };

        //---------------------------------
        // prepare fcp IO object
        // provide an interface to be called
        // by the application when it is initializing
        // objects that will be used after initial
        // html page loads
        //---------------------------------
        this.prepFCPIO = function() {
            // If the object is not ready, prepare the WSS FCP connection
            // and the data package object to use for FCP requests.
            if (this.ioState == fiIO_States.IONew) {
                if (this.fcpDataPkg == null) {
                    gInternalDataPkgID++;
                    if (gInternalDataPkgID > 9999) gInternalDataPkgID = 1;

                    this.fcpDataPkg = new fiFCP_DataPkg(gInternalDataPkgID);
                }

                this.ioState = fiIO_States.IOReady; // for now just mark successful
            }
        };

        //---------------------------------
        // Response Handler for FCP Request
        // This is the callback handler that is
        // invoked in response to a FCP WSS
        // load or store request
        //---------------------------------
        this.fcpResponseHandler = function(fcpDataPkg) {
            // NOTE:  this is a cbh (callback handler) and as
            // such the "this" object is not available!  use the var "me"

            // the data package responded should match our object's copy
            if (fcpDataPkg.internalID != me.fcpDataPkg.internalID) {
                // this is a pretty serious error... Something is not working properly
                alert("fcpResponseHandler fcpDataPkg Mismatch!!!");
            }

            // was the request successful?
            var requestResult = fcpDataPkg.getRequestResult();

            // if loading for each field returned need to notify the cbh
            // of the FCP Field Update
            if (requestResult == true) {
                // need to invoke the callback handler for the field update
                // for each field returned in the response package
                var nFields = fcpDataPkg.getNbrResponseFields();
                var FieldIndex;
                for (FieldIndex = 0; FieldIndex < nFields; FieldIndex++) {
                    var FieldKey = fcpDataPkg.getFieldKey(FieldIndex);
                    var FieldData = fcpDataPkg.getFieldValue(FieldIndex);

                    // notify the cbh for the field data update
                    if (me.cbhFieldUpdate != null) {
                        if (me.isListener == 1) {
                            var myDataInHex = fastecDecToHex(FieldData);
                            var myMsg =
                                "FCPIO NOTIFY Key[" +
                                FieldKey +
                                "] Data[0x" +
                                myDataInHex +
                                "]";
                            fastecLogTimeToConsole(gLogStatePolling, myMsg);
                        }
                        me.cbhFieldUpdate(me.isListener, FieldKey, FieldData);
                    }
                }
            }

            // setup the state of this object based on the response
            if (requestResult) me.ioState = fiIO_States.IOSuccess;
            else me.ioState = fiIO_States.IOError;

            // if there is a registered Callback Handler for when
            // the response is received make the call
            if (me.cbhRequestCompleted != null) {
                me.cbhRequestCompleted(requestResult);
            }

            // nothing is returned
        };

        //---------------------------------
        // Load FCP Data
        // main interface to service a load request
        // returns true if the request is in service or false if error occurred
        //---------------------------------
        this.fcpLoadRequest = function(
            inFieldsToLoad, // array of GIGE register values 1 or more
            incbhFieldUpdate, // cbh for the field update function
            incbhRequestCompleted // cbh when request is completed
        ) {
            var stateOfRequestOK = true; // assume true unless some error falls into place

            // if the IO Camera Object is in use can't continue
            if (this.ioState == fiIO_States.IOPending) stateOfRequestOK = false;

            // if this was never initialized
            if (stateOfRequestOK && this.ioState == fiIO_States.IONew)
                stateOfRequestOK = false;

            // if listener object can't call load on protocol set
            if (stateOfRequestOK && this.isListener == 1) {
                this.ioState = fiIO_States.IOSetupErr;
                stateOfRequestOK = false;
            }

            // if nothing to load can't continue
            if (stateOfRequestOK && inFieldsToLoad.length <= 0) {
                this.ioState = fiIO_States.IOSetupErr;
                stateOfRequestOK = false;
            }

            // reset this object to prepare for load
            if (stateOfRequestOK) {
                this.resetIO(true);

                // set up callbacks
                this.cbhFieldUpdate = incbhFieldUpdate;
                this.cbhRequestCompleted = incbhRequestCompleted;

                // prepare FCP Data Request Package Object
                if (this.fcpDataPkg.buildFCPRequest(0, inFieldsToLoad) != 1) {
                    // there is something wrong with the input array
                    this.ioState = fiIO_States.IOSetupErr;
                    stateOfRequestOK = false;
                }
            }

            // if here and all is ok send the request to the FCP WSS for servicing
            if (stateOfRequestOK) {
                // send the request through the WSS socket for the load
                var sentRequestCode = this.fcpWSS.send_request(
                    this.fcpDataPkg,
                    this.fcpResponseHandler
                );

                // don't have a connection available to mark the state of this object
                if (sentRequestCode == this.fcpWSS.sendNoConnection) {
                    // the request was sent without error
                    // mark IO object as waiting on pending request
                    this.ioState = fiIO_States.IONoWSSConn;
                    stateOfRequestOK = false;
                } else if (
                    sentRequestCode == this.fcpWSS.sendWaitingOnPastRequest
                ) {
                    // the request was sent without error
                    // mark IO object as waiting on pending request
                    this.ioState = fiIO_States.IOWSSBusy;
                    stateOfRequestOK = false;
                } else if (sentRequestCode == this.fcpWSS.sendSentRequest) {
                    // the request has been sent over the wire we are now pending.
                    this.ioState = fiIO_States.IOPending;
                    stateOfRequestOK = true;
                } else {
                    // we dont' know what this code represents...
                    this.ioState = fiIO_States.IOSetupErr;
                    stateOfRequestOK = false;
                }
            }

            // if the state of the request is in error need to callback the handler...
            if (!stateOfRequestOK) {
                if (incbhRequestCompleted !== null)
                    incbhRequestCompleted(stateOfRequestOK); // tell cbh this FAILED
            }

            // return the state of this IO request
            return stateOfRequestOK;
        };

        //---------------------------------
        // Store FCP Data
        // main interface to service a store request
        // returns true if the request is in service or
        // false if error occurs
        //---------------------------------
        this.fcpStoreRequest = function(
            inFieldsToStore, // array of GIGE register values 1 or more
            inDataToStore, // array of data values corresponding 1-1 with fields to store
            incbhFieldUpdate, // cbh for the field update function
            incbhRequestCompleted // cbh when request is completed
        ) {
            var stateOfRequestOK = true; // assume true unless some error falls into place

            // if the IO Camera Object is in use can't continue
            if (this.ioState == fiIO_States.IOPending) stateOfRequestOK = false;

            // if this was never initialized
            if (stateOfRequestOK && this.ioState == fiIO_States.IONew)
                stateOfRequestOK = false;

            // if listener object can't call load on protocol set
            if (stateOfRequestOK && this.isListener == 1) {
                this.ioState = fiIO_States.IOSetupErr;
                stateOfRequestOK = false;
            }

            // if nothing to store can't continue
            if (stateOfRequestOK && inFieldsToStore.length <= 0) {
                this.ioState = fiIO_States.IOSetupErr;
                stateOfRequestOK = false;
            }

            // if nothing to store can't continue
            if (stateOfRequestOK && inDataToStore.length <= 0) {
                this.ioState = fiIO_States.IOSetupErr;
                stateOfRequestOK = false;
            }

            // make sure size of the keys matches size of the values
            if (
                stateOfRequestOK &&
                inFieldsToStore.length != inDataToStore.length
            ) {
                this.ioState = fiIO_States.IOSetupErr;
                stateOfRequestOK = false;
            }

            // reset this object to prepare for load
            if (stateOfRequestOK) {
                this.resetIO(true);

                // set up callbacks
                this.cbhFieldUpdate = incbhFieldUpdate;
                this.cbhRequestCompleted = incbhRequestCompleted;

                // prepare FCP Data Request Package Object
                if (
                    this.fcpDataPkg.buildStoreRequest(
                        inFieldsToStore,
                        inDataToStore
                    ) != 1
                ) {
                    // there is something wrong with the input array
                    this.ioState = fiIO_States.IOSetupErr;
                    stateOfRequestOK = false;
                }
            }

            // if here and all is ok send the request to the FCP WSS for servicing
            if (stateOfRequestOK) {
                // send the request through the WSS socket for the load
                var sentRequestCode = this.fcpWSS.send_request(
                    this.fcpDataPkg,
                    this.fcpResponseHandler
                );

                // don't have a connection available to mark the state of this object
                if (sentRequestCode == this.fcpWSS.sendNoConnection) {
                    // the request was sent without error
                    // mark IO object as waiting on pending request
                    this.ioState = fiIO_States.IONoWSSConn;
                    stateOfRequestOK = false;
                } else if (
                    sentRequestCode == this.fcpWSS.sendWaitingOnPastRequest
                ) {
                    // the request was sent without error
                    // mark IO object as waiting on pending request
                    this.ioState = fiIO_States.IOWSSBusy;
                    stateOfRequestOK = false;
                } else if (sentRequestCode == this.fcpWSS.sendSentRequest) {
                    // the request has been sent over the wire we are now pending.
                    this.ioState = fiIO_States.IOPending;
                    stateOfRequestOK = true;
                } else {
                    // we dont' know what this code represents...
                    this.ioState = fiIO_States.IOSetupErr;
                    stateOfRequestOK = false;
                }
            }

            // if the state of the request is in error need to callback the handler...
            if (!stateOfRequestOK) {
                if (incbhRequestCompleted != null)
                    incbhRequestCompleted(stateOfRequestOK); // tell cbh this FAILED
            }
            // return the state of this IO request
            return stateOfRequestOK;
        };

        //---------------------------------
        // FCP Notification Listener
        // main interface to service a notify request
        // returns true if the request is in service or false if error occurred
        //---------------------------------
        this.fcpNotifyRequest = function(
            inFieldsToListenFor, // array of GIGE register values 1 or more
            incbhFieldUpdate, // cbh for the field update function
            incbhRequestCompleted // cbh when request is completed
        ) {
            var stateOfRequestOK = true; // assume true unless some error falls into place

            // if the IO Camera Object is in use can't continue
            if (this.ioState == fiIO_States.IOPending) stateOfRequestOK = false;

            // if this was never initialized
            if (stateOfRequestOK && this.ioState == fiIO_States.IONew)
                stateOfRequestOK = false;

            // if listener object can't call load on protocol set
            if (stateOfRequestOK && this.isListener != 1) {
                this.ioState = fiIO_States.IOSetupErr;
                stateOfRequestOK = false;
            }

            // if nothing to load can't continue
            if (stateOfRequestOK && inFieldsToListenFor.length <= 0) {
                this.ioState = fiIO_States.IOSetupErr;
                stateOfRequestOK = false;
            }

            // reset this object to prepare for load
            if (stateOfRequestOK) {
                this.resetIO(true);

                // set up callbacks
                this.cbhFieldUpdate = incbhFieldUpdate;
                this.cbhRequestCompleted = incbhRequestCompleted;

                // prepare FCP Data Request Package Object
                if (
                    this.fcpDataPkg.buildFCPRequest(1, inFieldsToListenFor) != 1
                ) {
                    // there is something wrong with the input array
                    this.ioState = fiIO_States.IOSetupErr;
                    stateOfRequestOK = false;
                }
            }

            // if here and all is ok send the request to the FCP WSS for servicing
            if (stateOfRequestOK) {
                // send the request through the WSS socket for the load
                var sentRequestCode = this.fcpWSS.send_request(
                    this.fcpDataPkg,
                    this.fcpResponseHandler
                );

                // don't have a connection available to mark the state of this object
                if (sentRequestCode == this.fcpWSS.sendNoConnection) {
                    // the request was sent without error
                    // mark IO object as waiting on pending request
                    this.ioState = fiIO_States.IONoWSSConn;
                    stateOfRequestOK = false;
                } else if (
                    sentRequestCode == this.fcpWSS.sendWaitingOnPastRequest
                ) {
                    // the request was sent without error
                    // mark IO object as waiting on pending request
                    this.ioState = fiIO_States.IOWSSBusy;
                    stateOfRequestOK = false;
                } else if (sentRequestCode == this.fcpWSS.sendSentRequest) {
                    // the request has been sent over the wire we are now pending.
                    this.ioState = fiIO_States.IOPending;
                    stateOfRequestOK = true;
                } else {
                    // we dont' know what this code represents...
                    this.ioState = fiIO_States.IOSetupErr;
                    stateOfRequestOK = false;
                }
            }

            // if the state of the request is in error need to callback the handler...
            if (!stateOfRequestOK) {
                if (incbhRequestCompleted != null)
                    incbhRequestCompleted(stateOfRequestOK); // tell cbh this FAILED
            }
            // return the state of this IO request
            return stateOfRequestOK;
        };
    } catch (err) {
        var theDetails = "caught: fiFCP_IO-" + err.description;
        gFaultHandler.logError(theDetails);
        return 0;
    }
}
