//------------------------------------------------
// fastec_fcp_data.js
//
// Copyright 2010-2018 (c) Fastec Imaging as an unpublished work.
// All Rights Reserved.
//
// The information contained herein is confidential property of
// Fastec Imaging. The use, copying, transfer or disclosure of such
// information is prohibited except by express written agreement
// with Fastec Imaging.
//
// Fastec Imaging Camera Control Web Application
// FCP Data Management for WSS Communications
//
// see gige_msg.js module for register data
//------------------------------------------------

//------------------------------------
// The Fastec FCP Data Package
// Responsible for managing a FCP data package
//
// Performs translation between JSON and Javascript
// using the fi-fcp-protocol format
//------------------------------------
function fiFCP_DataPkg(inMyUniqueDataID) {
    // The JSON request data being built.
    this.jsonRequestData = null;

    // The response data will be stored here.
    this.jsonResponseData = null;

    // Give a placeholder for a unique id to maintain integrity of
    // data across cbh in the fcp io stack journies
    this.internalID = inMyUniqueDataID;

    //---------------------------------------------------------------
    // This helper function determines if a fcp register string is
    // representing a valid Hexidecimal address - it doesn't validate
    // the value only whether it follows a FCP address format
    // returns 1 if ok or 0 if not
    //---------------------------------------------------------------
    this.isHexAddressValid = function(inHexStrToTest) {
        var hexStringOK = true;

        var fcpReg = "";
        fcpReg += inHexStrToTest;

        var testChar = "";

        // return 0 if length and start of string isn't right
        if (fcpReg.length != 6) return false;
        if (fcpReg.charAt(0) != "0") return false;
        testChar = fcpReg.charAt(1);
        if (testChar != "X" && testChar != "x") return false;

        var y;
        for (y = 2; y < 6 && hexStringOK == true; y++) {
            testChar = fcpReg.charAt(y);

            // check the range of characters acceptable for a HEX string
            if (
                (testChar >= "A" && testChar <= "F") ||
                (testChar >= "a" && testChar <= "f") ||
                (testChar >= "0" && testChar <= "9")
            )
                hexStringOK = true;
            else hexStringOK = false;
        }

        return hexStringOK;
    };

    //---------------------------------------------------------------
    // This function will intrepret the response package and return
    // a result of the request - true means successful, false means
    // incomplete or error
    //---------------------------------------------------------------
    this.getRequestResult = function() {
        var requestResult = false;

        // the package is completed without error ?
        if (this.jsonResponseData.pkginfo.status_code != undefined) {
            if (this.jsonResponseData.pkginfo.status_code == "1")
                requestResult = true;
        }

        return requestResult;
    };

    //---------------------------------------------------------------
    // This function will return whether or not the request was a
    // load request
    //---------------------------------------------------------------
    this.isLoadRequest = function() {
        var isLoadRequest = false;

        // if there is a valid response check the action key
        if (this.jsonResponseData != null) {
            if (this.jsonResponseData.action == "load") isLoadRequest = true;
        }

        return isLoadRequest;
    };

    //---------------------------------------------------------------
    // This function will return whether or not the request was a
    // store request
    //---------------------------------------------------------------
    this.isStoreRequest = function() {
        var isStoreRequest = false;

        // if there is a valid response check the action key
        if (this.jsonResponseData != null) {
            if (this.jsonResponseData.action == "store") isStoreRequest = true;
        }

        return isStoreRequest;
    };

    //---------------------------------------------------------------
    // This function will return whether or not the request was a
    // notify request
    //---------------------------------------------------------------
    this.isNotifyRequest = function() {
        var isNotifyRequest = false;

        // if there is a valid response check the action key
        if (this.jsonResponseData != null) {
            if (this.jsonResponseData.action == "notify")
                isNotifyRequest = true;
        }

        return isNotifyRequest;
    };

    //---------------------------------------------------------------
    // Returns the number of fields in the response package
    //---------------------------------------------------------------
    this.getNbrResponseFields = function() {
        var nFields = 0;

        if (this.jsonResponseData != null) {
            nFields = this.jsonResponseData.fields.length;
        }

        return nFields;
    };

    //---------------------------------------------------------------
    // Returns a field key from a response package
    //---------------------------------------------------------------
    this.getFieldKey = function(inFieldIndex) {
        var FieldKey = "";

        if (this.jsonResponseData != null) {
            if (
                inFieldIndex >= 0 &&
                inFieldIndex < this.getNbrResponseFields()
            ) {
                var myObj = this.jsonResponseData.fields[inFieldIndex];
                var myKey = Object.getOwnPropertyNames(myObj);

                // Check to see if the Field Key starts with the "0x"
                // if not add it.
                FieldKey = FieldKey + myKey;
                if (!this.isHexAddressValid(FieldKey)) {
                    FieldKey = "0x" + FieldKey;
                }
            }
        }

        return FieldKey;
    };

    //---------------------------------------------------------------
    // Returns a field value from a response package
    //---------------------------------------------------------------
    this.getFieldValue = function(inFieldIndex) {
        var FieldVal = null;
        var FieldKey = null;

        if (this.jsonResponseData != null) {
            if (
                inFieldIndex >= 0 &&
                inFieldIndex < this.getNbrResponseFields()
            ) {
                var myObj = this.jsonResponseData.fields[inFieldIndex];
                FieldKey = Object.getOwnPropertyNames(myObj);
                FieldVal = myObj[FieldKey];

                // probably should do some things by converting to a hex value
                // if a register?  TODO -- High Bit stuff plays into this

                // if this is a b64 encoded value decode it right here.
                // if the FieldVal has a b64_ as the first part of the string
                // decode it and recreate the data
                if (
                    FieldVal.charAt(0) == "b" &&
                    FieldVal.charAt(1) == "6" &&
                    FieldVal.charAt(2) == "4" &&
                    FieldVal.charAt(3) == "_"
                ) {
                    var length = FieldVal.length;
                    var todecode = FieldVal.slice(4, length - 4);
                    var decoded = atob(todecode);
                    FieldVal = decoded;
                }
            }
        }

        return FieldVal;
    };

    //---------------------------------------------------------------
    // This function will take in an array of FCP field names as
    // represented by the in_FieldArray and make sure they are valid
    // and set up the JSON load request string that will be used by
    // the fiFCP_IO object.
    //
    // It returns 1 if the JSON request data was successfully built or
    // 0 if there is an error with the input field data (syntax error)
    //---------------------------------------------------------------
    this.buildFCPRequest = function(
        in_Notifier, // 1 if notify 0 if load
        in_FieldArray // array of FCP registers to request
    ) {
        // reset the response data to empty string
        this.jsonResponseData = null;

        // build up the JSON request text used for a FCP load request
        // each valid FCP address will be the form of "0xYYYY" where
        // where the YYYY should be a valid Register address from gige
        // register set -- this will verify at a bare minimum the size
        // and the content of the register data to be within Hexidecimal
        // ranges
        if (in_Notifier) this.actionType = "notify";
        else this.actionType = "load";

        this.jsonRequestData = '{"action":"' + this.actionType + '","fields":['; // start of load request

        var fieldRequestError = "";
        var fieldSeperator = ",";

        // examples:
        //{"action":"load","fields":["0xE824","0x00E8"]}
        //{"action":"notify","fields":["0xE824","0x00E8"]}
        var nFields = in_FieldArray.length;
        var x;
        var lastFieldIndex = nFields - 1;

        // process each field requested for the load
        for (x = 0; x < nFields; x++) {
            var fcpReg = in_FieldArray[x];

            // does the fcp register requested follow valid hex format
            if (this.isHexAddressValid(fcpReg) === false) return 0;

            // valid field install it into the json request field
            //this.jsonRequestData = this.jsonRequestData + "{\"" + fcpReg + "\"}";
            this.jsonRequestData = this.jsonRequestData + '"' + fcpReg + '"';

            // more fields?
            if (x < lastFieldIndex)
                this.jsonRequestData = this.jsonRequestData + fieldSeperator;
            else this.jsonRequestData = this.jsonRequestData + "]}";
        }
        return 1;
    };

    //---------------------------------------------------------------
    // This function will take in an array of FCP field names as
    // represented by the in_FieldKeys and in_FieldValues and make
    // sure they are valid.  Sets up the JSON store request string
    // that will be used by the fiFCP_IO object.
    //
    // It returns 1 if the JSON request data was successfully built or
    // 0 if there is an error with the input field information which
    // is usually a (syntax error)
    //---------------------------------------------------------------
    this.buildStoreRequest = function(in_FieldKeys, in_FieldValues) {
        // reset the response data to empty string
        this.jsonResponseData = null;

        // build up the JSON request text used for a FCP store request
        // each valid FCP address will be the form of "0xYYYY" where
        // where the YYYY should be a valid Register address from gige
        // register set -- this will verify at a bare minimum the size
        // and the content of the register data to be within Hexidecimal
        // ranges
        this.jsonRequestData = '{"action":"store","fields":['; // start of store request

        var fieldRequestError = "";
        var fieldSeperator = ",";

        // example:
        //{"action":"store","fields":[{"0xE824","b64_*"},{"0x00E8","100"}]}
        var nFields = in_FieldKeys.length;
        var x;
        var lastFieldIndex = nFields - 1;

        // process each field requested for the load
        for (x = 0; x < nFields; x++) {
            var fcpReg = in_FieldKeys[x];
            var fcpData = in_FieldValues[x];

            // TODO:
            // does the value need to b64 encoding or will caller do this?

            // does the fcp register requested follow valid hex format
            if (this.isHexAddressValid(fcpReg) === false) return 0; // TODO:  Better error handling with info

            // valid field install it into the json request field
            //this.jsonRequestData = this.jsonRequestData + "{\"" + fcpReg + "\",\"" + fcpData + "\"}";
            this.jsonRequestData =
                this.jsonRequestData + '{"' + fcpReg + '":' + fcpData + "}";

            // more fields?
            if (x < lastFieldIndex)
                this.jsonRequestData = this.jsonRequestData + fieldSeperator;
            else this.jsonRequestData = this.jsonRequestData + "]}";
        }

        return 1;
    };
}

// end of fastec_fcp_data.js
