//------------------------------------------------
// fastec_camera.js
//
// Copyright 2010-2018 (c) Fastec Imaging as an unpublished work.
// All Rights Reserved.
//
// The information contained herein is confidential property of
// Fastec Imaging. The use, copying, transfer or disclosure of such
// information is prohibited except by express written agreement
// with Fastec Imaging.
//
// Fastec Imaging Camera Control Web Application
// Camera Data Management -- handles the data
// exchange and management of camera data for
// the connected camera over web interface.
//------------------------------------------------

//------------------------------------
// The camera data object.
//
// Responsible for providing UI data
// for all UI panels
//------------------------------------
function cameraData() {
    fastecLogToConsole(gLogTrace, "cameraData()");

    this.loadedCameraData = false; // We have not loaded camera data yet.
    this.loadingSysInfo = false; // Data object state when loading up the system data.

    //------------------------------------
    // SYSTEM INFORMATION PULLED FROM CAMERA
    //------------------------------------
    this.settingCameraNameTo = null;
    this.cameraName = "No camera attached"; // name for camera that is attached
    this.cameraModel = ""; // camera model name
    this.cameraSerialNo = ""; // camera seriel number
    this.cameraRev = ""; // camera revision
    this.cameraFPGARev = "";
    this.cameraSensorRev = "";
    this.cameraGIGERev = "";
    this.cameraBootLoadRev = "";

    // MAC address comes in parts - this will build state
    // to determine when full MAC address has been pulled.
    this.MAC_low = 0;
    this.MAC_high = 0;
    this.haveMAC = false;
    this.cameraMAC = ""; // The camera MAC address

    // ----PARTITION----
    this.partSize = 0; // how big is a partition
    this.partSizeRsvd = 0; // partition reserved space (metadata goes here)
    this.partSizeQuad = 0; // partition size in quadwords
    this.partSizeRsvdQuad = 0; // partition reserved space in quadwords

    this.actPartState; // Partition state for active partition.
    this.actPartNumFrames; // Partition's number of frames.
    this.actPartTrigFrame; // Partition's trigger frame number.
    this.actPartBitMode; // Partition's bit mode.

    // ---- AUTOSAVE ----
    this.autoSaveRegInHex = 0;
    this.autoSaveRegEnabled = false;
    this.autoSaveOlapEnabled = false;
    this.autoSaveSaving = false;
    this.autoSaveNeedsQueueUpdate = false;
    this.autoSaveCount = 0;
    this.autoSaveTotal = 0;
    this.autoSaveFileFmt = 0;
    this.autoSaveFileTag = "";
    this.autoSaveImageDest = 0;
    this.autoSaveImageDestUseCamName = false;
    this.autoSaveImageDestUseTag = false;
    this.autoSaveImageDestSaveXML = false;
    this.autoSaveImageDestMedia = "";
    this.autoSaveImageDestFmt = "";

    // ---- Long Record ----
    this.theLRModeRegInHex = 0;
    this.theLRModeEnabled = false;
    this.theLRModeDual = false;
    this.theLRModeROC = false;
    this.theLRModeBROC = false;
    this.theLRBROCLen = 0;
    this.theLRSecSizeRegInHex = 0;
    this.theLRSecSizeRegInDec = 0;
    this.theLRPartSizeInQuad = 0;
    this.theSSDBandWidthRegInHex = "0x0";
    this.theSSDBandWidth = 0;

    // Playback settings.
    this.pbEnabledRegValue = 0x0; // Holds the full register value from FCP
    this.pbEnabled = 0; // Is playback enabled (1) or disabled (0)
    this.pbBusy = false; // Is the playback control busy (true) or available (false)
    this.pbReverse = 0; // What direction forward (0) or reverse (1)
    this.pbCutInFrame = 0; // First frame in playback range
    this.pbCutOutFrame = 0; // Last frame in playback range
    this.pbStartFrame = 0; // Frame number to start playing at
    this.pbLastFrame = 0; // Read-only frame number where playback stopped
    this.pbLocalControlStream = 0; // Playback local control stream register
    this.pbRvwFrameNum = 0; // Playback frame number being reviewed
    this.pbFramePeriod = 0; // Playback ms rate

    // Recording settings.
    this.recFPS; // Framerate (frames/sec).
    this.recShutter; // shutter speed
    this.recOffsetX; // Offset X
    this.recOffsetY; // Offset Y
    this.recROIWidth; // ROI width
    this.recROIHeight; // ROI height
    this.recNumFrames; // CS can calc num frames in partition
    this.sysCameraState = gRegs.GIGE_CAMERA_STATE_IDLE;
    this.maxArmFrameNo = 0; // Used to keep track of highest frame number when armed.

    this.recMenuBinSample; // Record settings for Bin and Sample Low Word used for Menu
    this.recFullBinSample; // Record settings for Bin and Sample Full Register Value
    this.recBitDepth; // Record settings bit depth
    this.recSensorGain; // Record settings sensor gain
    this.recMaxSensorGain; // Maximum value for current sensor
    this.recNumFrames; // CS can calc num frames in partition
    this.recFPN; // Fixed pixel noise setting
    this.recFPNCalibrated = false; // Set using bit in FPN Register after loading it

    this.maxResWidth = 1280; // maximum width resolution for the sensor
    this.maxResHeight = 1024; // maximum height resolution for the sensor
    this.minROIWidth = 24; // minimum ROI width
    this.minROIHeight = 24; // minimum ROI height

    this.sys_cur_frame_no = 0; // Reported current frame number.
    this.sys_max_frame_rate = 0; // Maximum frame rate allowed. Zero means it is not set.
    this.sys_max_shutter = 0; // Maximum shutter speed allowed. Zero means it is not set.
    this.sys_min_shutter = 0; // Minimum shutter speed allowed. Zero means it is not set.
    this.sys_sensor_type = 0; // Zero means it is not set
    this.sysFrmOverhead = 6; // Frame overhead time default is 6

    var systemDataUIHandler = null; // Callback holder for loadSystemData(). When done updating settings, invoked the callback specified.

    //--------------------------------------
    // disable Autosave
    //--------------------------------------
    this.disableAutoSave = function(uiHandler) {
        // Get the current value of the register and clear the autosave enable bit.
        var autoSaveRegInDec = fastecHexToDec(gCameraData.autoSaveRegInHex);

        // Pull words in hex using string logic as when high bit is set it causes
        // havoc and cannot use bitwise operations as cannot circumvent sign bit.
        var loWord = csRegLoWord(autoSaveRegInDec);
        var hiWord = csRegHiWord(autoSaveRegInDec);

        // Enable bit is 1<<31 which is highest bit
        var bitStr = hiWord.substr(0, 1);
        bitStr = fastecHexToDec(bitStr);
        bitStr = bitStr & 0x7; // save all bits and clear enable flag (highest bit)

        var newAutoSaveReg = fastecDecToHex(bitStr);
        newAutoSaveReg = newAutoSaveReg.substr(1, 1) + hiWord.substr(1, 3); //pull rest of highword data
        newAutoSaveReg = newAutoSaveReg + loWord;

        // save as a hex string for the store process
        var asRegInHex = "0x" + newAutoSaveReg;
        this.writeRegister(gRegs.GIGE_REG_AUTOSAVE, asRegInHex, uiHandler);
    };

    //--------------------------------------
    // This Interface provides a way to notify
    // the Camera Object that it's internal data
    // has been updated.
    //
    // Currently this is called from a FCP Data
    // Request when the Response Package is received.
    //
    // This will be the callback handler (cbh) used
    // by the fiFCP_IO object.
    //
    // The cbh interface is expected to be in this format
    // boolean = cbhFieldUpdate(field_address, field_data_loaded).
    //--------------------------------------
    this.fcpFieldUpdate = function(
        inIsListener, // A '1' if the caller is FCP Listener.
        inFieldKey, // The FCP address as used in gige_msg.js.
        inFieldValue // The string data that was pulled for the field key.
    ) {
        fastecLogToConsole(gLogTrace, "fcpFieldUpdate()");

        // If the caller is a listener object (i.e. inIsListener == 1)
        // it means the change to the register is a notification.
        // If the camera has a registered cbh for the register in question,
        // that cbh will be invoked right now at time of change.
        var cbhFieldNotify = null; // Will be set if cbh is registered.
        if (inIsListener == 1) {
            cbhFieldNotify = gCameraData.fcpGetNotifyCBH(inFieldKey);
        }

        // Update the proper camera data object.
        switch (inFieldKey) {
            case gRegs.GIGE_REG_USER_DEF_NAME:
                gCameraData.cameraName = inFieldValue;
                break;
            case gRegs.GIGE_REG_MODEL_NAME:
                gCameraData.cameraModel = inFieldValue;
                break;
            case gRegs.GIGE_REG_SERIAL_NUM:
                gCameraData.cameraSerialNo = inFieldValue;
                break;
            case gRegs.GIGE_REG_FIRMWARE_VER:
                gCameraData.cameraRev = inFieldValue;
                break;
            case gRegs.GIGE_REG_FPGA_VER:
                gCameraData.cameraFPGARev = inFieldValue;
                break;
            case gRegs.GIGE_REG_GIGE_VER:
		gCameraData.handleGIGERev(inFieldValue);
                break;
            case gRegs.GIGE_REG_SENSOR_VER:
                gCameraData.cameraSensorRev = inFieldValue;
                break;
            case gRegs.GIGE_REG_BOOTLOADER_VER:
                gCameraData.cameraBootLoadRev = inFieldValue;
                break;
            case gRegs.GIGE_REG_MAC_LOW:
		gCameraData.handleMAC_low(inFieldValue);
                break;
            case gRegs.GIGE_REG_MAC_HIGH:
		gCameraData.handleMAC_high(inFieldValue);
                break;
            case gRegs.GIGE_REG_SENSOR_WIDTH:
                var newWidth = fastecHexToDec(fastecDecToHex(inFieldValue));
                if (gCameraData.maxResWidth != newWidth) gCameraData.maxResWidth = newWidth;
                break;
            case gRegs.GIGE_REG_SENSOR_HEIGHT:
                var newHeight = fastecHexToDec(fastecDecToHex(inFieldValue));
                if (gCameraData.maxResHeight != newHeight)
                    gCameraData.maxResHeight = newHeight;
                break;
            case gRegs.GIGE_REG_MIN_WIDTH:
                // Special case - if the min width is odd make it even
                // by going up by 1 instead of down which would be too small.
                var minWidth = fastecHexToDec(fastecDecToHex(inFieldValue));
                if (fastecIsOdd(minWidth)) minWidth = minWidth + 1;
                gCameraData.minROIWidth = minWidth;
                break;
            case gRegs.GIGE_REG_MIN_HEIGHT:
                // Special case - if the min height is odd make it even
                // by going up by 1 instead of down which would be too small.
                var minHeight = fastecHexToDec(fastecDecToHex(inFieldValue));
                if (fastecIsOdd(minHeight)) {
                    minHeight = minHeight + 1;
                }
                gCameraData.minROIHeight = minHeight;
                break;
            case gRegs.GIGE_REG_MAX_FRAME_RATE:
                gCameraData.sys_max_frame_rate = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_MAX_SHUTTER_SPEED:
                gCameraData.sys_max_shutter = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_MIN_SHUTTER_SPEED:
                gCameraData.sys_min_shutter = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_CAMERA_STATE:
                gCameraData.sysCameraState = fastecHexToDec(fastecDecToHex(inFieldValue));

                // Each time an armed state comes into play, reset the max incremental frame counter.
                if (gCameraData.sysCameraState == gRegs.GIGE_CAMERA_STATE_ARMED) {
                    gCameraData.maxArmFrameNo = 0;
                }
                break;
            case gRegs.GIGE_REG_PB_CONTROL:
		gCameraData.handlePBcontrol(inFieldValue);
                break;
            case gRegs.GIGE_REG_PB_STATUS:
                var csRegValueInHex = "0x" + fastecDecToHex(inFieldValue);

                if (csRegValueInHex & gRegs.GIGE_PBS_BUSY) {
		    gCameraData.pbBusy = true;
		} else {
		    gCameraData.pbBusy = false;
		}
                break;
            case gRegs.GIGE_REG_PB_CUT_IN_FRAME:
                gCameraData.pbCutInFrame = inFieldValue;
                break;
            case gRegs.GIGE_REG_PB_CUT_OUT_FRAME:
                gCameraData.pbCutOutFrame = inFieldValue;
                break;
            case gRegs.GIGE_REG_PB_START_FRAME:
                gCameraData.pbStartFrame = inFieldValue;
                break;
            case gRegs.GIGE_REG_PB_LAST_FRAME:
                gCameraData.pbLastFrame = inFieldValue;
                break;
            case gRegs.GIGE_REG_LOCALSTREAM_CONTROL:
                gCameraData.pbLocalControlStream = inFieldValue;
                break;
            case gRegs.GIGE_REG_CSI_RVW_FRAME_NUM:
                gCameraData.pbRvwFrameNum = inFieldValue;
		gVidReviewUI.handleReviewFrame(gCameraData.pbRvwFrameNum);
                break;
            case gRegs.GIGE_REG_PART_SIZE:
                // Pull out the number of bytes in total memory for a partition
                // the amount is in quadwords and a single quadword == 16 bytes.
                var partSizeInHex = fastecDecToHex(inFieldValue);
                gCameraData.partSizeQuad = fastecHexToDec(partSizeInHex);
                gCameraData.partSize = gCameraData.partSizeQuad * gCameraData.quadWordSize;
                break;
            case gRegs.GIGE_REG_PART_SIZE_RESERVED:
                // Pull out the number of bytes in total memory for a partition
                // the amount is in quadwords and a single quadword == 16 bytes.
                var partSizeRsvdInHex = fastecDecToHex(inFieldValue);
                gCameraData.partSizeRsvdQuad = fastecHexToDec(partSizeRsvdInHex);
                gCameraData.partSizeRsvd =
                    gCameraData.partSizeRsvdQuad * gCameraData.quadWordSize;
                break;
            case gRegs.GIGE_REG_PART_STATE:
                gCameraData.actPartState = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_PART_NUM_FRAMES:
                gCameraData.actPartNumFrames = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_PART_TRIG_FRAME:
                gCameraData.actPartTrigFrame = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_FRAME_RATE:
                gCameraData.recFPS = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_OFFSET_X:
                gCameraData.recOffsetX = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_OFFSET_Y:
                gCameraData.recOffsetY = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_WIDTH:
                gCameraData.recROIWidth = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_HEIGHT:
                gCameraData.recROIHeight = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_SHUTTER_SPEED:
                gCameraData.recShutter = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_NUM_FRAMES:
                gCameraData.recNumFrames = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_AUTOSAVE:
		gCameraData.handleAutoSave(inFieldValue);
                break;
            case gRegs.GIGE_REG_LTR_MODE:
		gCameraData.handleLTR(inFieldValue);
                break;
            case gRegs.GIGE_REG_BROC_LEN:
                gCameraData.theLRBROCLen = inFieldValue;
                break;
            case gRegs.GIGE_REG_LTR_SECTOR_COUNT:
                var csRegValue = inFieldValue;
                var csRegValueInHex = fastecDecToHex(inFieldValue);

                // Must be very careful to manipulate the bits properly
                // so save the original register in hex string for this purpose
                gCameraData.theLRSecSizeRegInHex = csRegValueInHex;
                gCameraData.theLRSecSizeRegInDec = csRegValue;
                gCameraData.theLRPartSizeInQuad = bytes_to_quadwords(csRegValue);
                break;
            case gRegs.GIGE_REG_SSD_BANDWIDTH:
                var csRegValue = inFieldValue;
                var csRegValueInHex = fastecDecToHex(inFieldValue);

                gCameraData.theSSDBandWidthRegInHex = "0x" + csRegValueInHex;
                gCameraData.theSSDBandWidth = csRegValue;
                break;
            case gRegs.GIGE_REG_BIT_MODE:
                gCameraData.recBitDepth = fastecHexToDec(fastecDecToHex(inFieldValue));
                break;
            case gRegs.GIGE_REG_SENSOR_OPTIONS:
                // Save the full register value for sensor options snapshot
                gCameraData.recFullBinSample = fastecHexToDec(fastecDecToHex(inFieldValue));

                // Pull only bin sample part
                gCameraData.recMenuBinSample = fastecHexToDec(
                    csRegLoWord(gCameraData.recFullBinSample)
                );
                break;
            case gRegs.GIGE_REG_SENSOR_TYPE:
                gCameraData.sys_sensor_type = fastecDecToHex(inFieldValue);
                break;
            case gRegs.GIGE_REG_CUR_FRAME_POS:
		gCameraData.handleCurFrame(inFieldValue);
                break;
	    case gRegs.GIGE_REG_TRIGGER_REQ_POS:
		gCameraData.handleTriggerReq(inFieldValue);
		break;

            default:
                fastecLogToConsole(
                    gLogTrace,
                    "fcpFieldUpdate(), Field: " +
                        inFieldKey +
                        " ignored.  Field Value[" +
                        inFieldValue +
                        "]"
                );
                break;
        }

        // If the callback handler for a notification is set, make the callback
        // here at the very end of all processing for field updates on the camera object.
        if (cbhFieldNotify != null) {
            cbhFieldNotify(inFieldKey);
        }
    };

    //--------------------------------------
    // Determines if a field has been registered
    // with a callback handler on notification
    // of a change to that field>
    //--------------------------------------
    this.fcpGetNotifyCBH = function(inFieldKey) {
        // the FCP address as used in gige_msg.js
        // Will return the CBH if one is associated with a field key
        // or null if not -- right now will default to calling a
        // general handler.
        return gCameraData.fcpNotifyDefaultCBH;
    };

    //--------------------------------------
    // A generic notify callback handler.
    // Use this until we can have individual FCP field handlers
    // available to process for register-specific notification.
    //--------------------------------------
    this.fcpNotifyDefaultCBH = function(inFieldKey) {
        // The FCP address as used in gige_msg.js
        // Right now just call the application state update logic.
        // The idea is that this will be replaced by specific
        // register->callback logic when it can be rewritten.
        // The argument is whether or not the load was ok - just send true
        gFastecAppMgr.onCameraInfoUpdate(true);
    };

    //--------------------------------------
    // Retrieve the max frames for the active partition.
    //--------------------------------------
    this.getMaxFrames = function(in_ROIWidth, in_ROIHeight) {
        // In order to call the frame count calculator, we need to have
        // the partition size in quad units.
        var part_size_quad = 1;

        // If in LR Mode a special partition size is pulled from GIGE_REG_LTR_SECTOR_COUNT
        if (gCameraData.theLRModeEnabled) {
            part_size_quad = gCameraData.theLRPartSizeInQuad; // use direct from register
	} else {
	    part_size_quad = gCameraData.getPartSizeQuadAvailable(); // handles reserved byte subtraction
	}

        // Call the calculator for the frame count.
        var frameCount = calc_frame_count(
            part_size_quad,
            in_ROIWidth,
            in_ROIHeight,
            gCameraData.recBitDepth,
            gCameraData.theLRModeEnabled
        );

        // Return our result.
        return frameCount;
    };

    //--------------------------------------
    // Get partition quadsize available for recording.
    // Note: This is dealing in quadwords not bytes.
    //--------------------------------------
    this.getPartSizeQuadAvailable = function() {
        var partSizeQuadAvailable = 0;
        if (this.partSizeQuad > 0) {
            partSizeQuadAvailable = this.partSizeQuad - this.partSizeRsvdQuad;
	}

        return partSizeQuadAvailable;
    };

    //--------------------------------------
    // Get recording time for the active
    // partition at current fps and ROI
    //--------------------------------------
    this.getRecTimeForActivePart = function(ROIW, ROIH, FPS) {
        var maxFrames = gCameraData.getMaxFrames(ROIW, ROIH);
        var recTimeInSec = 0;

        // If the frame rate is valid calculate the time.
        if (FPS > 0) recTimeInSec = maxFrames / FPS;

        return recTimeInSec;
    };

    //--------------------------------------
    // Determine Trigger Frame based on record settings
    //--------------------------------------
    this.getTriggerInfo = function() {
        var maxFrames = gCameraData.getMaxFrames(
            gCameraData.recROIWidth,
            gCameraData.recROIHeight
        );
        var maxTime = gCameraData.getRecTimeForActivePart(
            gCameraData.recROIWidth,
            gCameraData.recROIHeight,
            gCameraData.recFPS
        );

        // Should not happen - don't want math errors
        if (maxFrames <= 0) maxFrames = 1;
        if (maxTime <= 0) maxTime = 1;

        var triggerInfo = new Object();
        triggerInfo.pct = 0;
        triggerInfo.tFrame = 1;

        if (gCameraData.recTrigUsesFrames) {
            var trigFrame = gCameraData.recTrigPosition;
            if (trigFrame <= 0) {
		trigFrame = 1;
	    }

            triggerInfo.pct = trigFrame / maxFrames * 100;
            triggerInfo.pct = fastecMakeInteger(triggerInfo.pct); // make into integer percentage
            triggerInfo.tFrame = gCameraData.recTrigPosition;
        } else {
            triggerInfo.pct = gCameraData.recTrigPosition; // this is trigger percentage
            triggerInfo.tFrame = Math.floor(maxFrames * triggerInfo.pct / 100);
        }

        triggerInfo.maxFrames = maxFrames;
        triggerInfo.frmsBefore = triggerInfo.tFrame;
        triggerInfo.frmsAfter = maxFrames - triggerInfo.frmsBefore;

        // If there are frames after the trigger, allow for the trigger
	// frames's space in clip.
        if (triggerInfo.frmsAfter > 0) {
	    triggerInfo.frmsAfter = triggerInfo.frmsAfter - 1;
	}

        triggerInfo.timeBefore = maxTime * triggerInfo.pct;
        triggerInfo.timeAfter = maxTime - triggerInfo.timeBefore;

        return triggerInfo;
    };

    //---------------------------
    //---------------------------
    this.handleAutoSave = function(reg_value) {
	// (overlap mode and rapid fire capture data is here)
	var csRegValueInHex = fastecDecToHex(reg_value);

	// Must be very careful to manipulate the bits properly
	// so save the original register in hex string for this purpose
	gCameraData.autoSaveRegInHex = csRegValueInHex;

	// Is autosave enabled right now?
	if (reg_value & gRegs.GIGE_AUTOSAVE_ENABLE)
	    gCameraData.autoSaveRegEnabled = true;
	else gCameraData.autoSaveRegEnabled = false;

	// Is overlapped enabled?
	if (reg_value & gRegs.GIGE_AUTOSAVE_OVERLAP)
	    gCameraData.autoSaveOlapEnabled = true;
	else gCameraData.autoSaveOlapEnabled = false;

	// Is autosave saving right now?
	if (reg_value & gRegs.GIGE_AUTOSAVE_SAVING) gCameraData.autoSaveSaving = true;
	else gCameraData.autoSaveSaving = false;

	// Is autosave queue update needed right now?
	if (reg_value & gRegs.GIGE_AUTOSAVE_QUEUE_UPDATE)
	    gCameraData.autoSaveNeedsQueueUpdate = true;
	else gCameraData.autoSaveNeedsQueueUpdate = false;

	gCameraData.autoSaveCount = reg_value & gRegs.GIGE_AUTOSAVE_COUNT_MASK;
	gCameraData.autoSaveTotal =
	    (reg_value >> gRegs.GIGE_AUTOSAVE_TOTAL_SHIFT) &
	    gRegs.GIGE_AUTOSAVE_COUNT_MASK;
    };

    this.handleCurFrame = function(reg_value) {
	// Save the current frame number.
	gCameraData.sys_cur_frame_no = fastecHexToDec(fastecDecToHex(reg_value));

	// If camera is armed set arm position update UI trigger info
	if (gCameraData.sysCameraState == gRegs.GIGE_CAMERA_STATE_ARMED) {
	    // LR append mode is alittle different.
	    if (gCameraData.isCameraInLRAppendMode()) {
		gCameraData.maxArmFrameNo = gCameraData.sys_cur_frame_no;
	    } else {
		var triggerInfo = gCameraData.getTriggerInfo();

		// Save the highest current frame when armed until it wraps around.
		if (gCameraData.sys_cur_frame_no >= gCameraData.maxArmFrameNo) {
		    gCameraData.maxArmFrameNo = gCameraData.sys_cur_frame_no;
		} else {
		    // armed state has wrapped around again have a full buffer
		    gCameraData.maxArmFrameNo = triggerInfo.tFrame;
		}

		// if wrapped color the gauge yellow else blue
		if (gCameraData.maxArmFrameNo >= triggerInfo.tFrame) {
		    gArmTrigPctCtrl.setGaugeColor("Yellow"); // color of wrap around arm
		} else {
		    gArmTrigPctCtrl.setGaugeColor("Green"); // indicates wrap
		}

		// This SHOULD not happen but test for it anyway.
		if (gCameraData.maxArmFrameNo > triggerInfo.tFrame) {
		    gCameraData.maxArmFrameNo = triggerInfo.tFrame;
		}
	    }

	    var myMsg =
		"maxArmFrameNo - Armed [" +
		gCameraData.maxArmFrameNo +
		"] sysCur [" +
		gCameraData.sys_cur_frame_no +
		"] maxFrames[" +
		triggerInfo.maxFrames +
		"]";
	    fastecLogToConsole(gLogTriggerInfo, myMsg);
	} else if (gCameraData.sysCameraState == gRegs.GIGE_CAMERA_STATE_TRIGGERED) {
	    var triggerInfo = gCameraData.getTriggerInfo();

	    // if triggered set the max ArmFrameNo to go past trigger frame up to maximum
	    // Now that it's triggered add current frame to the trigger frame to get a max.
	    gCameraData.maxArmFrameNo = gCameraData.sys_cur_frame_no + triggerInfo.tFrame;

	    // This SHOULD not happen but test for it anyway.
	    if (gCameraData.maxArmFrameNo > triggerInfo.maxFrames) {
		gCameraData.maxArmFrameNo = triggerInfo.maxFrames;
	    }

	    var myMsg =
		"maxArmFrameNo - Trigger [" +
		gCameraData.maxArmFrameNo +
		"] sysCur [" +
		gCameraData.sys_cur_frame_no +
		"] maxFrames[" +
		triggerInfo.maxFrames +
		"]";
	    fastecLogToConsole(gLogTriggerInfo, myMsg);
	} else {
	    gCameraData.maxArmFrameNo = gCameraData.sys_cur_frame_no;
	    var myMsg =
		"maxArmFrameNo - Not Arm-Not Trigger[" +
		gCameraData.maxArmFrameNo +
		"]";
	    fastecLogToConsole(gLogTriggerInfo, myMsg);
	}
    };

    //---------------------------
    //---------------------------
    this.handleGIGERev = function(reg_value) {
	var gigeVerInHex = fastecDecToHex(reg_value);
	var major = fastecHexToDec(csRegHiWord(reg_value));
	var minor = fastecHexToDec(csRegLoWord(reg_value));
	var minorStr = "" + minor;

	if (minorStr.length < 2) {
	    minorStr = "0" + minorStr;
	}

	gCameraData.cameraGIGERev = major + "." + minorStr;
    };


    //---------------------------
    //---------------------------
    this.handleLTR = function(reg_value) {
	var csRegValueInHex = fastecDecToHex(reg_value);

	// Must be very careful to manipulate the bits properly
	// so save the original register in hex string for this purpose
	gCameraData.theLRModeRegInHex = csRegValueInHex;

	// Is LR mode enabled right now?
	if (reg_value & gRegs.GIGE_LTR_MODE_ENABLE)
	    gCameraData.theLRModeEnabled = true;
	else gCameraData.theLRModeEnabled = false;

	// Is overlapped enabled?
	if (reg_value & gRegs.GIGE_LTR_MODE_DUAL) gCameraData.theLRModeDual = true;
	else gCameraData.theLRModeDual = false;

	// Is ROC enabled?
	if (reg_value & gRegs.GIGE_LTR_MODE_ROC) gCameraData.theLRModeROC = true;
	else gCameraData.theLRModeROC = false;

	// Is BROC enabled?
	if (reg_value & gRegs.GIGE_LTR_MODE_BROC) gCameraData.theLRModeBROC = true;
	else gCameraData.theLRModeBROC = false;
    };

    //---------------------------
    //---------------------------
    this.handleMAC_low = function(reg_value) {
	gCameraData.MAC_low = reg_value;

	if (gCameraData.MAC_high != 0) {
	    gCameraData.cameraMAC = formatMACString(
		gCameraData.MAC_low,
		gCameraData.MAC_high
	    );

	    gCameraData.haveMAC = true;
	}
    };

    //---------------------------
    //---------------------------
    this.handleMAC_high = function(reg_value) {
	gCameraData.MAC_high = reg_value;
	if (gCameraData.MAC_low != 0) {
	    gCameraData.cameraMAC = formatMACString(
		gCameraData.MAC_low,
		gCameraData.MAC_high
	    );

	    gCameraData.haveMAC = true;
	}
    };

    //---------------------------
    //---------------------------
    this.handlePBcontrol = function(reg_value) {
	var csRegValueInHex = "0x" + fastecDecToHex(reg_value);

	// Store the full state of the register value to preserve
	// the upper 12 bits -- they are used by the FPGA/Camera Server
	gCameraData.pbEnabledRegValue = csRegValueInHex;

	// Is the enable bit set?
	if ((csRegValueInHex & gRegs.GIGE_PBC_ENABLE) == gRegs.GIGE_PBC_ENABLE) {
	    gCameraData.pbEnabled = true;
	} else {
	    gCameraData.pbEnabled = false;
	}

	// Is the reverse bit set?
	if ((csRegValueInHex & gRegs.GIGE_PBC_REVERSE) == gRegs.GIGE_PBC_REVERSE) {
	    gCameraData.pbReverse = true;
	} else {
	    gCameraData.pbReverse = false;
	}
    };

    //---------------------------
    //---------------------------
    this.handleTriggerReq = function(reg_value) {
	var theValue = fastecHexToDec(fastecDecToHex(reg_value));
	var theValInHex = fastecDecToHex(theValue);
	var theValueAndFrames = theValue - gRegs.GIGE_TRIGGER_REQ_FRAME_MASK;
	if (theValueAndFrames >= 0)
	{
	    gCameraData.recTrigUsesFrames = true;
	    gCameraData.recTrigPosition = theValueAndFrames;
	}
	else
	{
	    gCameraData.recTrigUsesFrames = false;
	    gCameraData.recTrigPosition = theValue;
	}

	// If the camera is in an LR Append Mode force a
	// setting (the camera really should do this!)
	if (gCameraData.isCameraInLRAppendMode())
	{
	    // force to use percent with a 0% value
	    gCameraData.recTrigUsesFrames = false;
	    gCameraData.recTrigPosition = 0;
	}
    };

    
    //---------------------------
    // hasCaptureData will
    // test the state to see if camera has
    // data captured in the buffer
    //---------------------------
    this.hasCaptureData = function() {
        if (
            this.actPartState == gRegs.GIGE_PART_STATE_COMPLETE ||
            this.actPartState == gRegs.GIGE_PART_STATE_LOCKED
        ) {
            return true;
        }

        return false;
    };

    //--------------------------------------
    // Is this a LINCE sensor?
    //--------------------------------------
    this.isLINCESensor = function() {
        if (
            gCameraData.sys_sensor_type == gLINCE5Color ||
            gCameraData.sys_sensor_type == gLINCE5MMono
        ) {
            return true;
        }

        return false; // not a LINCE sensor.
    };

    //--------------------------------------
    // Is the Camera in a Long Record Append Mode (ROC or BROC)
    //--------------------------------------
    this.isCameraInLRAppendMode = function() {
        if (gCameraData.theLRModeEnabled) {
            if (gCameraData.theLRModeROC || gCameraData.theLRModeBROC) return true;
        }

        return false; // not in ROC or BROC LR Append Mode
    };

    //--------------------------------------
    // Is the state of the camera in review
    // with recorded partition?
    //--------------------------------------
    this.isReviewPartitionActive = function() {
	if (gCameraData.sysCameraState == gRegs.GIGE_CAMERA_STATE_REVIEW) {
	    // Only support if the partition is in complete or locked state
	    if ((gCameraData.actPartState == gRegs.GIGE_PART_STATE_COMPLETE) ||
		(gCameraData.actPartState == gRegs.GIGE_PART_STATE_LOCKED))
	    {
		return true;
	    }
	}

	// The camera is not in review state with a recorded partition so return false.
	return false;
    };

    //--------------------------------------
    // Load System Data Part I using FCP WSS
    //--------------------------------------
    this.loadSystemData = function(uiHandler) {
        fastecLogToConsole(gLogTrace, "loadSystemData()");

        // If already running don't try to do this again
        if (this.loadingSysInfo) return;

        // Pull the FCP data using the WSS
        var fcpFieldsToLoad = new Array();

        systemDataUIHandler = uiHandler; // store since will chain loads
        this.loadingSysInfo = true;

        // Read up our registers from the camera.
        fcpFieldsToLoad.push(gRegs.GIGE_REG_USER_DEF_NAME);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MODEL_NAME);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_FIRMWARE_VER);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_FPGA_VER);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_SENSOR_VER);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_GIGE_VER);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MAC_LOW);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MAC_HIGH);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MEM_SIZE);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MEM_SIZE_RESERVED);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_SESSION_LENGTH);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MIN_WIDTH);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MIN_HEIGHT);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MAX_FRAME_RATE);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MAX_SHUTTER_SPEED);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MIN_SHUTTER_SPEED);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_SENSOR_TYPE);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_CAMERA_STATE);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_AVAIL_MEDIA);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PREFERENCES);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_TEMPERATURE);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_WIDTH);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_HEIGHT);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_OFFSET_X);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_OFFSET_Y);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_FRAME_RATE);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_SHUTTER_SPEED);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_SENSOR_OPTIONS);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_BIT_MODE);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_NUM_FRAMES);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_TRIGGER_REQ_POS);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_LTR_MODE);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_BROC_LEN);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_LTR_SECTOR_COUNT);

        // invoke the FCP Load Request
        gFCP_IO.fcpLoadRequest(
            fcpFieldsToLoad,
            gCameraData.fcpFieldUpdate, // field update handler
            gCameraData.loadSystemDataPartII
        ); // cbh to update UI after response
    };

    //--------------------------------------
    // Load System Data Part II
    //--------------------------------------
    this.loadSystemDataPartII = function(loadedOK) {
        fastecLogToConsole(gLogTrace, "loadSystemDataPartII(" + loadedOK + ")");

        // Pull the FCP data using the WSS
        var fcpFieldsToLoad = new Array();

        fcpFieldsToLoad.push(gRegs.GIGE_REG_PART_NUM_FRAMES);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PART_TRIG_FRAME);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PART_STATE);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PART_SIZE);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PART_SIZE_RESERVED);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_SENSOR_WIDTH);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_SENSOR_HEIGHT);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_MAX_PART);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_NUM_PART);

        // invoke the FCP Load Request
        gFCP_IO.fcpLoadRequest(
            fcpFieldsToLoad,
            gCameraData.fcpFieldUpdate, // field update handler
            gCameraData.loadSystemDataFinish
        ); // cbh to update UI after response
    };

    //--------------------------------------
    // Load System Data is complete.
    //--------------------------------------
    this.loadSystemDataFinish = function(loadedOK) {
        fastecLogToConsole(gLogTrace, "loadSystemDataFinish(" + loadedOK + ")");

        // All done loading.
	if (systemDataUIHandler != null) {
	    systemDataUIHandler(loadedOK);
	    systemDataUIHandler = null;
	}
    };

    //--------------------------------------
    // Return the maximum frame rate for current camera settings.
    //--------------------------------------
    this.max_exposure = function() {
        return gCameraData.sys_max_shutter;
    };

    //--------------------------------------
    // Return the maximum frame rate for current camera settings.
    //--------------------------------------
    this.max_framerate = function() {
        var max_frame_rate = calc_max_framerate(
            gCameraData.recOffsetX,
            gCameraData.recROIWidth,
            gCameraData.recROIHeight,
            gCameraData.recBitDepth,
            gCameraData.recFullBinSample,
            gCameraData.theLRModeEnabled,
            gCameraData.theSSDBandWidth
        );

        return max_frame_rate;
    };

    //--------------------------------------
    // Set the exposure of the camera.
    //--------------------------------------
    this.setCameraExposure = function(new_exposure, uiHandler) {
        // Write out the new shutter speed (aka "exposure").
        gCameraData.writeRegister(gRegs.GIGE_REG_SHUTTER_SPEED, new_exposure, uiHandler);
    };

    //--------------------------------------
    // Set the framerate of the camera.
    //--------------------------------------
    this.setCameraFramerate = function(new_framerate, uiHandler) {
        // Store the FCP data using the WSS.
        var fcpFieldKeys = new Array();
        var fcpFieldValues = new Array();

        fcpFieldKeys.push(gRegs.GIGE_REG_HOLD);
        fcpFieldValues.push("1");
        fcpFieldKeys.push(gRegs.GIGE_REG_FRAME_RATE);
        fcpFieldValues.push(new_framerate);
        fcpFieldKeys.push(gRegs.GIGE_REG_HOLD);
        fcpFieldValues.push("0");

        // Invoke the FCP store request
        gFCP_IO.fcpStoreRequest(
            fcpFieldKeys, // the array of field keys to store
            fcpFieldValues, // the array of field values to store
            gCameraData.fcpStoreFieldUpdate, // field update handler
            uiHandler
        ); // cbh to update UI after response
    };

    //--------------------------------------
    // Set the ROI of the camera.
    //--------------------------------------
    this.setCameraROI = function(new_width, new_height, new_x, new_y, uiHandler) {
        // Store the FCP data using the WSS.
        var fcpFieldKeys = new Array();
        var fcpFieldValues = new Array();

        fcpFieldKeys.push(gRegs.GIGE_REG_HOLD);
        fcpFieldValues.push("1");
        fcpFieldKeys.push(gRegs.GIGE_REG_WIDTH);
        fcpFieldValues.push(new_width);
        fcpFieldKeys.push(gRegs.GIGE_REG_HEIGHT);
        fcpFieldValues.push(new_height);
        fcpFieldKeys.push(gRegs.GIGE_REG_OFFSET_X);
        fcpFieldValues.push(new_x);
        fcpFieldKeys.push(gRegs.GIGE_REG_OFFSET_Y);
        fcpFieldValues.push(new_y);
        fcpFieldKeys.push(gRegs.GIGE_REG_HOLD);
        fcpFieldValues.push("0");

        // Invoke the FCP store request
        gFCP_IO.fcpStoreRequest(
            fcpFieldKeys, // the array of field keys to store
            fcpFieldValues, // the array of field values to store
            gCameraData.fcpStoreFieldUpdate, // field update handler
            uiHandler
        ); // cbh to update UI after response
    };

    //--------------------------------------
    // Set the state of the camera.
    //--------------------------------------
    this.setCameraState = function(newState, uiHandler) {
        fastecLogToConsole(gLogTrace, "setCameraState(" + newState + ")");
        this.writeRegister(gRegs.GIGE_REG_CHANGE_CAMERA_STATE, newState, uiHandler);
    };

    //--------------------------------------
    // Setup which FCP registers are in a notification
    // watch list -- when changed these will come
    // through the gCameraData.fcpFieldUpdate function
    // on a field basis.
    //
    // The cbhResponseResult is used for reporting response results.
    //--------------------------------------
    this.setupCameraNotifications = function(cbhResponseResult) {
        // Pull the FCP data using the WSS
        var fcpFieldsToWatch = new Array();

        fcpFieldsToWatch.push(gRegs.GIGE_REG_CAMERA_STATE);

        // Partition information.
        fcpFieldsToWatch.push(gRegs.GIGE_REG_PART_SIZE);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_PART_NUM_FRAMES);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_PART_TRIG_FRAME);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_PART_STATE);

        fcpFieldsToWatch.push(gRegs.GIGE_REG_CUR_FRAME_POS);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_TRIGGER_REQ_POS);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_FRAME_RATE);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_MAX_SHUTTER_SPEED);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_AUTOSAVE);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_LTR_MODE);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_BROC_LEN);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_PB_CONTROL);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_PB_STATUS);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_PB_START_FRAME);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_CSI_RVW_FRAME_NUM);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_SENSOR_WIDTH);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_SENSOR_HEIGHT);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_WIDTH);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_HEIGHT);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_OFFSET_X);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_OFFSET_Y);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_SHUTTER_SPEED);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_SENSOR_OPTIONS);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_BIT_MODE);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_NUM_FRAMES);
        fcpFieldsToWatch.push(gRegs.GIGE_REG_LOCALSTREAM_CONTROL);

        // Invoke the FCP Load Request
        if (!gFCP_StateListener.fcpNotifyRequest(
                fcpFieldsToWatch, // The array of fields to watch.
                gCameraData.fcpFieldUpdate, // Field update handler.
                cbhResponseResult
	    )
	) {
            // Callback handler to update UI after response.
            // For now just alert - but should report the exact nature of the issue
            alert("Unable to setup the FCP State Listener!");
        }
    };

    this.fcpStoreFieldUpdate = function(
        inIsListener, // A '1' if the caller is FCP Listener.
        inFieldKey, // The FCP address as used in gige_msg.js.
        inFieldValue // The string data that was pulled for the field key.
    ) {
	var myMsg = "fcpStoreFieldUpdate(" + inIsListener + ", " +
	    inFieldKey + ", " + inFieldValue + ")";
        fastecLogToConsole(gLogAll, myMsg);
    };

    this.updateReviewRegs = function(uiHandler) {
        fastecLogToConsole(gLogTrace, "updateReviewRegs");

	// Callback when we're done with the loads.
        systemDataUIHandler = uiHandler;

        // Pull the FCP data using the WSS
        var fcpFieldsToLoad = new Array();

        fcpFieldsToLoad.push(gRegs.GIGE_REG_PB_CONTROL);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PB_STATUS);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PB_FRAME_PERIOD);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PB_CUT_IN_FRAME);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PB_CUT_OUT_FRAME);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PB_START_FRAME);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PB_LAST_FRAME);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PART_TRIG_FRAME);
        fcpFieldsToLoad.push(gRegs.GIGE_REG_PART_NUM_FRAMES);

        // invoke the FCP Load Request
        gFCP_IO.fcpLoadRequest(
            fcpFieldsToLoad,
            gCameraData.fcpFieldUpdate, // field update handler
            gCameraData.loadSystemDataFinish
        ); // cbh to update UI after response
    };

    //--------------------------------------
    // Method to write registers to the camera.
    //--------------------------------------
    this.writeRegister = function(regAddr, regValue, uiHandler) {
        // Store the FCP data using the WSS.
        var fcpFieldKeys = new Array();
        var fcpFieldValues = new Array();

        fcpFieldKeys.push(regAddr);
        fcpFieldValues.push(regValue);

        // Invoke the FCP store request
        gFCP_IO.fcpStoreRequest(
            fcpFieldKeys, // the array of field keys to store
            fcpFieldValues, // the array of field values to store
            gCameraData.fcpStoreFieldUpdate, // field update handler
            uiHandler
        ); // cbh to update UI after response
    };
}
